// Build 1756477359
// Build 1756344610
// Build 1756156973
// Build 1756100226
// Cache buster: 1755793756
// Cache buster: 1754966000
/**
 * CarbonGuru Browser Extension - Simplified Content Script
 * Only injects floating analyze button, no popup functionality
 */

/**
 * Format cost with smart decimal precision
 * 4 decimals for sub-cent amounts (< $0.01), 2 decimals otherwise
 * @param {number|string} cost - The cost value to format
 * @returns {string} Formatted cost string with $ prefix
 */
const formatCost = (cost) => {
    const amount = parseFloat(cost) || 0;
    if (amount <= 0) return '$0.00';
    if (amount < 0.01) return '$' + amount.toFixed(4);
    return '$' + amount.toFixed(2);
};

void 0;

class CarbonGuruContentScript {
    constructor() {
        this.analyzeButton = null;
        this.cartDetector = null;

        // Environment configuration loaded from background script
        this.environmentConfig = null;

        // Prevent duplicate dialog coordination
        this.duplicateDialogShown = false;
        this.duplicateDialogTimestamp = 0;
        this.duplicateDialogResetTimeout = 30000; // 30 seconds

        // Track initialization state for timing issues
        this.isInitialized = false;
        this.initializationPromise = null;

        // Track cart processing session to hide buttons during individual item analysis
        this.cartProcessingActive = false;

        // Domain info for status messages (populated from validate_url response)
        this.currentDomainInfo = null;

        // Gmail integration
        this.gmailDetector = null;
        this.gmailIntegration = null;
        this.gmailFloatingTile = null;
        this.isGmailPage = false;

        // Status message queue - ensures each status message is visible for minimum time
        this.statusMessageQueue = [];
        this.statusMessageDisplaying = false;
        this.statusMessageMinDisplayMs = 1500; // 1.5 seconds minimum display per message
        this.statusMessageLastDisplayTime = 0;

        this.init();
    }

    /**
     * Get the source name prefix for status messages.
     * Returns friendly name for known sources, "new source" for unknown domains.
     */
    getSourcePrefix() {
        if (this.currentDomainInfo &&
            (this.currentDomainInfo.pattern_type === 'builtin' ||
             this.currentDomainInfo.pattern_type === 'learned')) {
            return this.currentDomainInfo.source_name || 'this source';
        }
        return 'new source';
    }

    /**
     * Get the item type descriptor (cart or page) based on domain info.
     */
    getItemType() {
        if (this.currentDomainInfo && this.currentDomainInfo.is_cart_url) {
            return 'cart';
        }
        return 'page';
    }

    /**
     * Queue a status message render function for display.
     * Ensures each status message is visible for minimum time before next one.
     * @param {Function} renderFn - Function that renders the status message
     * @param {string} messageType - Type of message for logging (optional)
     * @param {boolean} isPriority - If true, clears queue and shows immediately (for final results)
     */
    queueStatusMessage(renderFn, messageType = 'status', isPriority = false) {
        if (isPriority) {
            // Priority messages (like final results) clear the queue and show immediately
            this.statusMessageQueue = [];
            this.statusMessageDisplaying = false;
            void 0;
            renderFn();
            this.statusMessageLastDisplayTime = Date.now();
            return;
        }

        // Add to queue
        this.statusMessageQueue.push({ renderFn, messageType, queuedAt: Date.now() });
        void 0;

        // Start processing if not already
        if (!this.statusMessageDisplaying) {
            this.processStatusQueue();
        }
    }

    /**
     * Process the status message queue, ensuring minimum display time between messages.
     */
    processStatusQueue() {
        if (this.statusMessageQueue.length === 0) {
            this.statusMessageDisplaying = false;
            return;
        }

        this.statusMessageDisplaying = true;

        // Calculate time since last message was displayed
        const timeSinceLastDisplay = Date.now() - this.statusMessageLastDisplayTime;
        const timeToWait = Math.max(0, this.statusMessageMinDisplayMs - timeSinceLastDisplay);

        if (timeToWait > 0 && this.statusMessageLastDisplayTime > 0) {
            // Wait before showing next message
            void 0;
            setTimeout(() => this.showNextStatusMessage(), timeToWait);
        } else {
            // Show immediately
            this.showNextStatusMessage();
        }
    }

    /**
     * Show the next status message from the queue.
     */
    showNextStatusMessage() {
        if (this.statusMessageQueue.length === 0) {
            this.statusMessageDisplaying = false;
            return;
        }

        const { renderFn, messageType, queuedAt } = this.statusMessageQueue.shift();
        const waitTime = Date.now() - queuedAt;
        void 0;

        // Render the message
        try {
            renderFn();
        } catch (error) {
            console.error(`CarbonGuru: Error rendering status message (${messageType}):`, error);
        }

        // Update display time
        this.statusMessageLastDisplayTime = Date.now();

        // Schedule next message if queue has more
        if (this.statusMessageQueue.length > 0) {
            setTimeout(() => this.processStatusQueue(), this.statusMessageMinDisplayMs);
        } else {
            this.statusMessageDisplaying = false;
        }
    }

    /**
     * Clear the status message queue (useful when analysis completes or is cancelled).
     */
    clearStatusQueue() {
        const cleared = this.statusMessageQueue.length;
        this.statusMessageQueue = [];
        this.statusMessageDisplaying = false;
        if (cleared > 0) {
            void 0;
        }
    }

    async loadEnvironmentConfig() {
        return new Promise((resolve) => {
            browser.runtime.sendMessage({action: 'getEnvironmentConfig'}, (response) => {
                this.environmentConfig = response;
                void 0;
                resolve();
            });
        });
    }

    async getBackendUrl() {
        if (!this.environmentConfig) {
            await this.loadEnvironmentConfig();
        }
        return this.environmentConfig.backendUrl;
    }

    async getProductUrl(path) {
        return new Promise((resolve) => {
            browser.runtime.sendMessage({
                action: 'getProductUrl', 
                path: path
            }, (response) => {
                resolve(response.url);
            });
        });
    }

    async getCartSummaryUrl(sessionId, analysisType = null) {
        return new Promise((resolve) => {
            browser.runtime.sendMessage({
                action: 'getCartSummaryUrl',
                sessionId: sessionId,
                analysisType: analysisType
            }, (response) => {
                resolve(response.url);
            });
        });
    }

    async getOffsetUrl(sessionId) {
        return new Promise((resolve) => {
            browser.runtime.sendMessage({
                action: 'getOffsetUrl', 
                sessionId: sessionId
            }, (response) => {
                resolve(response.url);
            });
        });
    }

    async getReviewUrl(detailsUrl) {
        return new Promise((resolve) => {
            browser.runtime.sendMessage({
                action: 'getReviewUrl', 
                detailsUrl: detailsUrl
            }, (response) => {
                resolve(response.url);
            });
        });
    }

    async init() {
        void 0;

        // Load environment configuration first
        await this.loadEnvironmentConfig();

        // Initialize Gmail detection if on Gmail page
        if (typeof GmailDetector !== 'undefined') {
            this.gmailDetector = new GmailDetector();
            const gmailPageInfo = this.gmailDetector.detectPage();
            this.isGmailPage = gmailPageInfo.isGmail;

            if (this.isGmailPage) {
                void 0;

                // Initialize Gmail integration
                if (typeof GmailIntegration !== 'undefined') {
                    this.gmailIntegration = new GmailIntegration(
                        this.environmentConfig.backendUrl,
                        () => this.getAuthHeaderForGmail()
                    );
                }

                // Initialize Gmail floating tile extension
                if (typeof FloatingTileGmailExtension !== 'undefined') {
                    this.gmailFloatingTile = new FloatingTileGmailExtension(this);
                    await this.gmailFloatingTile.init(this.gmailDetector, this.gmailIntegration);
                }
            }
        }

        void 0;
        this.createAnalyzeButton();
        this.setupMessageListener();
        try {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            runtime.sendMessage({ action: 'getVersionInfo' }, (resp) => {
                if (resp && resp.currentVersion) {
                    void 0;
                }
            });
        } catch (_) {}

        // Mark as initialized - ready to receive auth notifications
        this.isInitialized = true;
        void 0;
    }

    // Wait for initialization to complete before processing messages
    async waitForInitialization() {
        if (this.isInitialized) {
            return Promise.resolve();
        }

        // Create a promise that resolves when initialized
        if (!this.initializationPromise) {
            this.initializationPromise = new Promise((resolve) => {
                const checkInit = setInterval(() => {
                    if (this.isInitialized) {
                        clearInterval(checkInit);
                        resolve();
                    }
                }, 50); // Check every 50ms

                // Timeout after 5 seconds
                setTimeout(() => {
                    clearInterval(checkInit);
                    void 0;
                    resolve();
                }, 5000);
            });
        }

        return this.initializationPromise;
    }


    createAnalyzeButton() {
        // Remove existing button if it exists
        if (this.analyzeButton) {
            this.analyzeButton.remove();
        }

        this.analyzeButton = document.createElement('div');
        this.analyzeButton.className = 'carbon-guru-analyze-button';
        this.analyzeButton.innerHTML = `
            <div id="carbon-guru-tile" class="carbon-tile" role="region" aria-label="CarbonGuru carbon footprint">
                <div id="carbon-tile-header" class="carbon-tile-header">
            <button id="carbon-analyze-btn" class="carbon-guru-floating-btn" title="Get Carbon Footprint with CarbonGuru">
                <svg class="cg-logo-icon" width="18" height="18" viewBox="0 0 100 100" fill="currentColor" xmlns="http://www.w3.org/2000/svg" aria-hidden="true"><path fill-rule="evenodd" clip-rule="evenodd" d="M57.712,53.028c-0.123,2.017-1.382,4.043-1.382,4.043c0.023-0.7-0.689-2.96-0.689-2.96c-0.847,0.328-2.271,0.297-2.271,0.297s0.509,1.292,0.494,2.861c-0.015,1.567-2.071,7.892-2.071,7.892c-2.889-5.056-2.368-9.47-2.368-9.47c-4.357,0.11-8.089-1.086-8.089-1.086l0.888,3.354c-4.196-1.593-4.34-4.241-4.34-4.241l-2.17-0.592c0.396,1.935,3.477,5.494,4.834,6.609c1.356,1.114,4.245,4.007,5.919,6.115c1.674,2.111,1.851,3.641,2.071,5.229c0.22,1.588-0.799,7.372-0.789,8.386c0.01,1.013-1.506,9.358-2.171,10.654c-0.664,1.294-3.453,2.564-3.453,2.564s2.101,0.69,4.341,0.69c2.24,0,7.628-1.086,8.977-1.086c1.35,0,2.048,0.39,3.749,0.396c1.701,0.005,2.071-0.396,2.071-0.396c-4.691-1.703-4.734-2.269-4.734-2.269c-1.539-2.703-2.167-9.948-1.283-19.139c0.885-9.189,7.695-14.797,7.695-14.797C59.347,54.936,57.712,53.028,57.712,53.028z M86.812,33.297c4.475-20.846-13.907-19.039-13.907-19.039C65.606,1.896,46.071,6.859,46.071,6.859c-21.148-9.305-22.788,7.103-22.788,7.103c-17.239-1.287-9.569,16.278-9.569,16.278c-9.458,16.146,4.637,15.586,4.637,15.586c1.916,12.708,16.572,5.229,16.572,5.229C48.416,56.259,58.5,50.856,58.5,50.856c13.719,11.396,20.223,0,20.223,0C94.372,51.65,86.812,33.297,86.812,33.297z"/></svg>
                <span class="cg-btn-text">Get Carbon Footprint</span>
            </button>
                    <button id="carbon-tile-toggle" class="carbon-tile-toggle" title="Expand" aria-expanded="false" aria-controls="carbon-tile-content" aria-label="Toggle details panel">▼</button>
                </div>
                <div id="carbon-tile-content" class="carbon-tile-content" style="display: none;">
                    <div id="carbon-status-section" class="carbon-status-section" style="display: none;">
                        <div id="carbon-current-page" class="carbon-current-page" style="display: block;">
                            <div id="carbon-page-url" class="carbon-page-url" style="font-size: 11px; color: #666; margin-bottom: 4px;"></div>
                            <div id="carbon-page-analysis-type" class="carbon-page-analysis-type">Page type: Unknown</div>
                            <div id="carbon-document-type" class="carbon-document-type" style="display: none; font-size: 12px; color: #333; margin-top: 4px;"></div>
                        </div>
                        <div id="carbon-status-icon" class="carbon-status-icon"></div>
                        <div id="carbon-status-text" class="carbon-status-text" aria-live="polite" aria-atomic="true">Ready to analyze carbon footprint...</div>
                        <div id="carbon-progress-bar" class="carbon-progress-bar" style="display: none;">
                            <div class="carbon-progress-fill"></div>
                        </div>
                        <div id="carbon-status-buttons" class="carbon-status-buttons" style="display: none;"></div>
                    </div>
                    <div id="carbon-results-section" class="carbon-results-section" style="display: none;">
                        <div class="carbon-result-header">✅ Analysis Complete</div>
                        <div id="carbon-footprint-display" class="carbon-footprint-display"></div>
                        <div id="carbon-product-details" class="carbon-product-details"></div>
                        <div id="carbon-pcf-comparison-display" class="carbon-pcf-comparison-display"></div>
                        <div class="carbon-action-buttons">
                            <div class="carbon-action-buttons-row">
                                <button id="carbon-offset-btn" class="carbon-offset-btn" style="display: none;">
                                    <span class="btn-label">Offset:</span>
                                    <span class="btn-cost">...</span>
                                </button>
                                <button id="carbon-removal-btn" class="carbon-removal-btn" style="display: none;">
                                    <span class="btn-label">Remove:</span>
                                    <span class="btn-cost">...</span>
                                </button>
                            </div>
                            <!-- DISABLED: View Details button - product name is now linked instead
                            <button id="carbon-view-details-btn" class="carbon-view-details-btn" style="display: none;">View details</button>
                            -->
                        </div>
                    </div>
                    <div id="carbon-recent-section" class="carbon-recent-section">
                        <div class="carbon-recent-header">Your Carbon Footprints</div>
                        <div id="carbon-recent-items" class="carbon-recent-items">
                            <div class="carbon-loading">Loading recent items...</div>
                        </div>
                    </div>
                </div>
            </div>
        `;

        document.body.appendChild(this.analyzeButton);
        this.setupAnalyzeButtonListener();
        
        // Setup toggle listener after DOM insertion
        setTimeout(() => {
            this.setupTileToggleListener();
            this.setupFloatingTileBasketListener();

            // Setup minimal state hover listeners
            const tileHeader = document.getElementById('carbon-tile-header');
            if (tileHeader) {
                tileHeader.addEventListener('mouseenter', () => {
                    if (this.isInMinimalState()) {
                        this.expandFromMinimal();
                    } else {
                        // Reset timer on any hover even when not minimal
                        this.startMinimalStateTimer();
                    }
                });

                tileHeader.addEventListener('focusin', () => {
                    if (this.isInMinimalState()) {
                        this.expandFromMinimal();
                    } else {
                        this.startMinimalStateTimer();
                    }
                });
            }
        }, 100);

        // Load recent items initially
        this.loadRecentItems();

        // Update page analysis type
        this.updatePageAnalysisType();

        // Restore tile state from previous session (keep open across page refreshes)
        this.restoreTileState();

        // On CarbonGuru domains, start in minimal (icon-only) state immediately
        if (window.location.hostname.toLowerCase().includes('carbonguru.io')) {
            // Apply classes synchronously before paint to avoid full-size flash
            const btn = document.getElementById('carbon-analyze-btn');
            const header = document.getElementById('carbon-tile-header');
            const tile = document.getElementById('carbon-guru-tile');
            if (btn) btn.classList.add('cg-minimal');
            if (header) header.classList.add('cg-header-minimal');
            if (tile) tile.classList.add('cg-tile-minimal');
        }

        // Start the minimal-state auto-collapse timer
        this.startMinimalStateTimer();
    }

    setupAnalyzeButtonListener() {
        const analyzeBtn = this.analyzeButton.querySelector('#carbon-analyze-btn');

        void 0;

        if (analyzeBtn) {
            analyzeBtn.addEventListener('click', () => {
                void 0;
                void 0;
                this.startAnalysis();
            });
            void 0;
        } else {
            console.error('CarbonGuru: ❌ Failed to find analyze button - listener not attached!');
        }
    }

    async extractFullPage() {
        try {
            void 0;
            void 0;

            // Show initial status (will be updated with source name after validation)
            this.showStatus('Analyzing...', {
                type: 'info'
            });

            // Check for existing mappings and handle user choice
            const currentUrl = window.location.href;
            const validationResponse = await this.validateUrl(currentUrl);

            // Store domain_info for use in all status messages
            this.currentDomainInfo = validationResponse.domain_info || null;
            void 0;

            // Build source-aware status message
            const source = this.getSourcePrefix();
            const itemType = this.getItemType();

            // Flag to track if user already made a choice about duplicates
            let userChoiceHandled = false;

            if (!validationResponse.success) {
                void 0;
                this.showStatus(`Analyzing ${source} ${itemType}...`, {
                    type: 'info'
                    // No duration - will be replaced by progress message
                });
                // Continue with analysis despite validation failure
            } else if (validationResponse.has_existing_items && validationResponse.existing_items && validationResponse.existing_items.length > 0) {
                void 0;
                
                // Show user choice dialog for existing mappings
                const userChoice = await this.showDeduplicationNotification(validationResponse.existing_items, currentUrl);
                userChoiceHandled = true;
                
                if (userChoice === 'use_existing') {
                    void 0;
                    
                    // Get the first existing item to display
                    const selectedItem = validationResponse.existing_items[0];
                    
                    try {
                        // Copy the existing mapping (this function displays the results internally)
                        await this.copyExistingMapping(selectedItem, currentUrl);

                        // NOTE: copyExistingMapping() already calls showResults() with the complete API response data
                        // including PCF comparison and social cost. DO NOT call showResults() again here as it would
                        // overwrite the display with incomplete data from the validation response.

                        return; // Exit early - no need to continue with analysis
                        
                    } catch (error) {
                        console.error('CarbonGuru: Error handling existing mapping:', error);
                        this.showStatus('Error using existing mapping. Proceeding with new analysis...', {
                            type: 'warning',
                            duration: 3000
                        });
                    }
                } else {
                    void 0;
                }
            } else {
                void 0;
                this.showStatus(`Analyzing ${source} ${itemType}...`, {
                    type: 'info'
                });
            }
            
            // Step 2: Capture the full page content from the current page
            // Uses smart wait - only delays if JavaScript framework placeholders are detected
            const fullPageContent = await this.captureFullPageContent();

            // Detect Amazon anti-bot interstitial only on Amazon domains
            try {
                const hostname = (window.location.hostname || '').toLowerCase();
                if (hostname.includes('amazon.')) {
                    const htmlLower = (fullPageContent && fullPageContent.full_page_content) ? fullPageContent.full_page_content.toLowerCase() : '';
                    if (htmlLower.includes('click the button below to continue shopping') || htmlLower.includes('robot check') || htmlLower.includes('captcha')) {
                        this.showStatus('Amazon is asking for a verification. Click "Continue shopping" on the page, then click Analyze again.', { type: 'warning', duration: 6000 });
                        void 0;
                        return;
                    }
                }
            } catch (e) {
                // Non-fatal
            }
            
            if (!fullPageContent.success) {
                throw new Error(fullPageContent.error);
            }
            
            void 0;
            
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;

            // Send message to background with retry to avoid transient "Receiving end does not exist"
            const sendWithRetry = async (payload, maxAttempts = 3) => {
                let lastError = null;
                for (let attempt = 1; attempt <= maxAttempts; attempt++) {
                    try {
                        // eslint-disable-next-line no-await-in-loop
                        const resp = await runtime.sendMessage(payload);
                        return resp;
                    } catch (err) {
                        lastError = err;
                        const message = (err && err.message) ? err.message : String(err);
                        void 0;
                        if (message && message.toLowerCase().includes('receiving end does not exist')) {
                            // Give background a moment to initialize
                            // eslint-disable-next-line no-await-in-loop
                            await new Promise(r => setTimeout(r, 500));
                            continue;
                        }
                        break;
                    }
                }
                throw lastError || new Error('sendMessage failed');
            };

            // Skip duplicate check if user already made a choice
            if (!userChoiceHandled) {
                void 0;
                
                // OPTIMIZATION: Use the existing validationResponse instead of making another call
                if (validationResponse.has_existing_items && validationResponse.existing_items.length > 0) {
                    void 0;
                    
                    // Show user choice dialog for existing mappings
                    const userChoice = await this.showDeduplicationNotification(validationResponse.existing_items, currentUrl);
                    
                    if (userChoice === 'use_existing') {
                        void 0;
                        
                        // Get the first existing item to display
                        const selectedItem = validationResponse.existing_items[0];
                        
                        try {
                            // Copy the existing mapping (this function displays the results internally)
                            await this.copyExistingMapping(selectedItem, currentUrl);

                            // NOTE: copyExistingMapping() already calls showResults() with the complete API response data
                            // including PCF comparison and social cost. DO NOT call showResults() again here as it would
                            // overwrite the display with incomplete data from the validation response.

                        } catch (error) {
                            console.error('CarbonGuru: Error handling existing mapping:', error);
                            this.showStatus('Error using existing mapping', { type: 'error' });
                        }
                        return;
                    } else {
                        void 0;
                        // Continue with processing below
                    }
                }
            } else {
                void 0;
            }

            // Start keepalive to prevent Chrome MV3 service worker termination during analysis
            if (window.cgKeepalive) {
                window.cgKeepalive.start();
            }

            const response = await sendWithRetry({
                action: 'processProductWithExtraction',
                data: {
                    url: currentUrl,
                    action: 'analyze',
                    full_page_content: fullPageContent.full_page_content,
                    page_title: fullPageContent.page_title,
                    extracted_at: fullPageContent.extracted_at,
                    extraction_method: fullPageContent.extraction_method
                }
            });
            if (!response || !response.success) {
                console.error('CarbonGuru: Background script processing failed:', response);
                // Show error to user instead of duplicating processing
                this.showStatus(`Error: ${response?.error || 'Could not process page'}`, 'error');
                return;
            }
            
            void 0;
            
            if (response.success) {
                void 0;
                // Analysis will show progress and results in the unified status area
            } else {
                if (response.isDuplicate) {
                    void 0;
                    this.showStatus('Analysis already in progress for this page.', {
                        type: 'warning',
                        duration: 4000
                    });
                } else {
                    console.error('CarbonGuru: Extraction failed:', response.error);
                    this.showStatus(`Analysis failed: ${response.error}`, {
                        type: 'error',
                        duration: 4000
                    });
                }
            }
        } catch (error) {
            console.error('CarbonGuru: Error extracting full page:', error);
            // Stop keepalive on error
            if (window.cgKeepalive) {
                window.cgKeepalive.stop();
            }
            // Don't show error for service worker timeouts - recovery happens automatically
            if (error.message?.includes('service worker')) {
                void 0;
                return;
            }
            this.showStatus(`Analysis failed: ${error.message || 'Unknown error'}`, {
                type: 'error',
                duration: 6000
            });
        }
    }

    /**
     * Check if HTML contains unrendered JavaScript framework placeholders.
     * Returns true if placeholders are detected that indicate the page isn't fully rendered.
     */
    hasUnrenderedPlaceholders(html) {
        // TJX sites (Marshalls, TJMaxx, HomeGoods) use ???ada.xxx??? placeholders
        if (/\?\?\?ada\.[a-zA-Z]+\?\?\?/.test(html)) {
            void 0;
            return true;
        }

        // Generic AngularJS/Angular placeholders: {{...}}
        // Only flag if there are multiple or in price-related context
        const angularMatches = html.match(/\{\{[^}]+\}\}/g);
        if (angularMatches && angularMatches.length > 3) {
            void 0;
            return true;
        }

        // Vue.js placeholders in price elements
        if (/class="[^"]*price[^"]*"[^>]*>\s*\{\{/.test(html)) {
            void 0;
            return true;
        }

        return false;
    }

    /**
     * Wait for JavaScript framework to finish rendering the page.
     * Uses HTML size stability detection to handle Angular/Vue/React sites.
     * @param {number} maxWaitMs - Maximum time to wait in milliseconds
     * @param {number} checkIntervalMs - How often to check for rendering completion
     * @returns {Promise<string>} - The captured HTML
     */
    async waitForRenderedContent(maxWaitMs = 10000, checkIntervalMs = 200) {
        const startTime = Date.now();
        let lastHtml = document.documentElement.outerHTML;
        let lastHtmlLength = lastHtml.length;
        let stableCount = 0;
        const STABILITY_THRESHOLD = 1000; // Consider stable if growth < 1KB
        const STABLE_CHECKS_REQUIRED = 3; // Require 3 consecutive stable checks

        // Check if this is a known dynamic site that needs waiting
        const hostname = window.location.hostname.toLowerCase();
        const pathname = window.location.pathname.toLowerCase();

        // Sites with async cart loading via API (Amazon-owned, React/Redux sites)
        const isAsyncCartSite = hostname.includes('zappos.') ||
                               hostname.includes('6pm.') ||  // Zappos sister site
                               hostname.includes('endless.');  // Another Amazon site

        // Cart pages on async sites need extra time for API data to load
        const isCartPage = pathname.includes('/cart') ||
                          pathname.includes('/bag') ||
                          pathname.includes('/basket');

        const isDynamicSite = hostname.includes('tjx.com') ||
                             hostname.includes('marshalls.') ||
                             hostname.includes('tjmaxx.') ||
                             hostname.includes('homegoods.') ||
                             hostname.includes('sierratradingpost.') ||
                             isAsyncCartSite ||
                             this.hasUnrenderedPlaceholders(lastHtml);

        // For static sites with no placeholders, use quick stability check
        if (!isDynamicSite) {
            void 0;
            maxWaitMs = 2000; // Shorter timeout for static sites
        } else if (isAsyncCartSite && isCartPage) {
            // Async cart sites need extra time for API to load cart items
            maxWaitMs = 5000; // 5 seconds for cart pages on async sites
            void 0;
        } else {
            void 0;
        }

        void 0;

        // For async cart sites, also wait for cart items to appear in DOM
        const needsCartItemCheck = isAsyncCartSite && isCartPage;

        // Wait for HTML size to stabilize
        while (Date.now() - startTime < maxWaitMs) {
            await new Promise(resolve => setTimeout(resolve, checkIntervalMs));

            const currentHtml = document.documentElement.outerHTML;
            const currentLength = currentHtml.length;
            const growth = currentLength - lastHtmlLength;

            // Check if HTML size is stable (not growing significantly)
            if (growth < STABILITY_THRESHOLD) {
                stableCount++;
                void 0;

                // Also check if placeholders are now resolved
                if (!this.hasUnrenderedPlaceholders(currentHtml)) {
                    if (stableCount >= STABLE_CHECKS_REQUIRED) {
                        void 0;
                        return currentHtml;
                    }
                } else if (stableCount >= STABLE_CHECKS_REQUIRED + 2) {
                    // Even with placeholders, if content is very stable, capture it
                    void 0;
                    return currentHtml;
                }
            } else {
                // Content still growing, reset stability counter
                stableCount = 0;
                void 0;
            }

            lastHtml = currentHtml;
            lastHtmlLength = currentLength;
        }

        const finalHtml = document.documentElement.outerHTML;
        void 0;
        return finalHtml;
    }

    async captureFullPageContent() {
        try {
            // Wait for JavaScript frameworks to finish rendering if needed
            const fullHTML = await this.waitForRenderedContent();

            const pageContent = {
                success: true,
                full_page_content: fullHTML,
                page_title: document.title,
                extracted_at: new Date().toISOString(),
                extraction_method: 'content_script_capture'
            };

            void 0;
            return pageContent;

        } catch (error) {
            console.error('CarbonGuru: Error capturing page content:', error);
            return {
                success: false,
                error: error.message
            };
        }
    }

    setupMessageListener() {
        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
        runtime.onMessage.addListener((message, sender, sendResponse) => {
            void 0;

            if (message.action === 'ping') {
                // Respond to ping requests - used to check if content script is ready
                sendResponse({
                    pong: true,
                    initialized: this.isInitialized || false,
                    timestamp: Date.now()
                });
                return false; // Sync response
            }

            if (message.action === 'authenticationStatusChanged') {
                // Handle authentication status change notification
                void 0;

                // Wait for initialization before handling auth change
                this.waitForInitialization().then(() => {
                    void 0;
                    this.handleAuthenticationChange(message.isAuthenticated, message.user);
                    sendResponse({ received: true });
                });
                return true; // Keep message channel open for async response
            } else if (message.action === 'captureFullPageContent') {
                this.captureFullPageContent().then(sendResponse);
                return true; // Keep message channel open for async response
            } else if (message.action === 'showExistingMappingsDialog') {
                // Handle request to show user choice dialog for existing mappings
                void 0;
                const isCart = message.isCart || false;
                this.showDeduplicationNotification(message.existingItems, message.url, isCart).then(choice => {
                    sendResponse({ choice: choice });
                });
                return true; // Keep message channel open for async response
            } else if (message.action === 'analysisStatus') {
                // Handle intermediate status updates (cart extraction status, etc.)
                void 0;
                if (message.success && message.data) {
                    void 0;

                    // Display status message immediately (cart extraction results)
                    // Include both shopping_cart and order_page as cart-like results (multi-item processing)
                    // Check both top-level and structured_content for analysis_type
                    const analysisType = message.data.analysis_type || (message.data.structured_content && message.data.structured_content.analysis_type);
                    const isCartResult = analysisType === 'shopping_cart' || analysisType === 'order_page' || message.data.type === 'cart';
                    const contentType = message.data.content_type;

                    // Handle single product status messages (text extraction and cleaning)
                    // Use queue to ensure each message is visible for minimum time
                    if (contentType === 'text_extraction_status') {
                        // Stage: Original text description extracted
                        void 0;
                        const data = message.data;
                        this.queueStatusMessage(() => this.showTextExtractionStatus(data), 'text_extraction_status');
                    } else if (contentType === 'text_cleaning_status') {
                        // Stage: Cleaned text description processed
                        void 0;
                        const data = message.data;
                        this.queueStatusMessage(() => this.showTextCleaningStatus(data), 'text_cleaning_status');
                    } else if (contentType === 'product_extraction_status') {
                        // Stage: Product info extracted from LLM
                        void 0;
                        const data = message.data;
                        this.queueStatusMessage(() => this.showProductExtractionStatus(data), 'product_extraction_status');
                    } else if (contentType === 'learning_triggered_status') {
                        // Stage: Learning triggered for new domain
                        void 0;
                        const data = message.data;
                        this.queueStatusMessage(() => this.showLearningTriggeredStatus(data), 'learning_triggered_status');
                    } else if (contentType === 'individual_item_complete' && !isCartResult) {
                        // PRODUCT FINAL RESULT: For single products, individual_item_complete is the final Stage
                        // Extract product data from structured_content.current_item
                        void 0;
                        const structuredContent = message.data.structured_content || message.data;
                        const currentItem = structuredContent.current_item || message.data;

                        // Transform current_item to match analysisComplete data format
                        const productData = {
                            success: true,
                            product_carbon_footprint: currentItem.product_carbon_footprint,
                            product_name: currentItem.product_name,
                            price: currentItem.price,
                            currency: currentItem.currency,
                            sector_code: currentItem.sector_code,
                            sector_name: currentItem.sector_name,
                            country_code: currentItem.country_code,
                            pcf_comparison: currentItem.pcf_comparison,
                            social_cost_usd: currentItem.social_cost_usd,
                            offset_cost_usd: currentItem.offset_cost_usd ?? currentItem.display_cost_usd ?? currentItem.orderable_cost_usd,
                            display_cost_usd: currentItem.display_cost_usd,
                            orderable_cost_usd: currentItem.orderable_cost_usd,
                            analysis_type: 'individual_product',
                            type: 'product'
                        };

                        // Show final results with PCF comparison (priority - clears queue and shows immediately)
                        this.clearStatusQueue();
                        this.showAnalysisComplete(productData);
                    } else if (isCartResult) {
                        // Cart-specific status messages
                        // Check structured_content first (backend stores cart data there)
                        void 0;

                        const structuredData = message.data.structured_content || message.data;
                        void 0;

                        const items = structuredData.items || message.data.items || [];
                        void 0;

                        const totalItems = structuredData.item_count || message.data.total_items || 0;
                        void 0;

                        void 0;

                        if (contentType === 'cart_extraction_status') {
                            // Stage 1: Cart extracted - show status_message with items list
                            // Use message.data (has properly mapped fields from background.js)
                            // NOT structured_content (raw ActivityLog data missing enhanced_display_title)
                            const extractionData = message.data;
                            this.showCartExtractionStatus(extractionData);
                        } else if (contentType === 'individual_item_complete') {
                            // Stage 2: Individual item completion - show current item
                            this.showCartItemProgress(message.data);
                        } else if (items.length > 0 && items.length < totalItems) {
                            // Stage 2 fallback: Items processing - show items with individual footprints
                            this.showCartItemProgress(message.data);
                        } else if (items.length > 0) {
                            // All items processed but not marked complete yet - show progress
                            this.showCartItemProgress(message.data);
                        }
                        // Stage 3 (Analysis Complete) handled by analysisComplete action
                    }
                }
                sendResponse({ success: true });
            } else if (message.action === 'analysisComplete') {
                // Handle completion of background analysis - show results
                void 0;
                // Stop keepalive - analysis is complete
                if (window.cgKeepalive) {
                    window.cgKeepalive.stop();
                }
                // Clear any pending status messages - final results take priority
                this.clearStatusQueue();
                if (message.success && message.data) {
                    void 0;

                    // For carts & orders: show status message only (no expandable tile)
                    // For products: show expandable tile with results
                    const isCartResult = message.data.analysis_type === 'shopping_cart' || message.data.type === 'cart';
                    const isOrderResult = message.data.analysis_type === 'order_page' || message.data.type === 'order';
                    const contentType = message.data.content_type;

                    // Check if this is an individual cart item (Stage 2) vs final summary (Stage 3)
                    if ((isCartResult || isOrderResult) && contentType === 'individual_item_complete') {
                        // Individual cart/order item: show green tile with hideButtons logic
                        this.showResults(message.data).then(() => {
                            setTimeout(() => this.loadRecentItems(), 2000);
                        });
                    } else if ((isCartResult || isOrderResult) && contentType === 'analysis_complete') {
                        // Final cart/order summary: BOTH use showCartCompletionStatus (same display)
                        this.showCartCompletionStatus(message.data);
                        setTimeout(() => this.loadRecentItems(), 2000);
                    } else if (isCartResult || isOrderResult) {
                        // Fallback for other cart/order messages
                        this.showCartCompletionStatus(message.data);
                        setTimeout(() => this.loadRecentItems(), 2000);
                    } else {
                        // Product: show expandable tile with results
                        this.showResults(message.data).then(() => {
                            setTimeout(() => this.loadRecentItems(), 2000);
                        });
                    }
                } else {
                    void 0;
                    this.hideProgress();
                    this.showStatus(`Analysis failed: ${message.error || message.data?.error || 'Unknown error'}`, {
                        type: 'error',
                        duration: 6000
                    });
                }
                sendResponse({ success: true });
            } else if (message.action === 'showUnifiedNotification') {
                // Handle notification from background script
                void 0;
                this.showStatus(message.message, message.options || {});
                sendResponse({ success: true });
            } else if (message.action === 'updateWooCommerceUrl') {
                // DISABLED: View Details button no longer shown - product name is linked instead
                void 0;

                // Refresh Recent Items to update any internal URLs with WooCommerce URLs
                void 0;
                this.loadRecentItems();

                sendResponse({ success: true });
            } else if (message.action === 'showAuthenticationRequired') {
                // Handle authentication required notification
                void 0;
                this.showAuthenticationPrompt(message.message, message.url);
                sendResponse({ success: true });
            }
        });
    }

    // ========== UNIFIED STATUS SYSTEM ==========
    
    showStatus(message, options = {}) {
        const {
            type = 'info',
            icon = null,
            showProgress = false,
            duration = 0,
            interactive = false,
            buttons = [],
            existingItems = [],
            timeout = null,
            onTimeout = null,
            additionalInfo = null
        } = options;

        let statusSection = document.getElementById('carbon-status-section');

        if (!statusSection) {
            console.error('CarbonGuru: Status section not found - likely in Gmail popout window');
            return;
        }

        // DEFENSIVE FIX: Check if status elements exist, restore if missing
        let statusIcon = document.getElementById('carbon-status-icon');
        let statusText = document.getElementById('carbon-status-text');
        let progressBar = document.getElementById('carbon-progress-bar');
        let statusButtons = document.getElementById('carbon-status-buttons');
        const currentPageDiv = document.getElementById('carbon-current-page');

        // If critical elements are missing, restore the status section structure
        if (!statusIcon || !statusText) {
            void 0;

            // Save any existing content
            const existingContent = statusSection.innerHTML;

            // Restore basic status structure
            statusSection.innerHTML = `
                <div id="carbon-status-icon" class="carbon-status-icon"></div>
                <div id="carbon-status-text" class="carbon-status-text"></div>
                <div id="carbon-progress-bar" class="carbon-progress-bar" style="display: none;">
                    <div class="carbon-progress-fill" style="width: 0%;"></div>
                </div>
                <div id="carbon-status-buttons" class="carbon-status-buttons"></div>
            `;

            // Re-query the restored elements
            statusIcon = document.getElementById('carbon-status-icon');
            statusText = document.getElementById('carbon-status-text');
            progressBar = document.getElementById('carbon-progress-bar');
            statusButtons = document.getElementById('carbon-status-buttons');

            void 0;
        }

        // Clear any existing timeouts/intervals
        if (statusSection._statusTimeout) {
            clearTimeout(statusSection._statusTimeout);
        }
        if (statusSection._timeoutInterval) {
            clearInterval(statusSection._timeoutInterval);
        }

        // Show the status section
        statusSection.style.display = 'block';
        statusSection.className = `carbon-status-section carbon-status-${type}`;

        // Show current page if tile is open and no analysis in progress
        // Hide current page div for 'progress' and 'warning' (remap UI) states per brand design
        if (currentPageDiv && type !== 'progress' && type !== 'warning') {
            const currentUrl = window.location.href;
            const pageUrlDiv = document.getElementById('carbon-page-url');
            if (pageUrlDiv) {
                pageUrlDiv.textContent = `Current URL: ${currentUrl}`;
                currentPageDiv.style.display = 'block';
            }
        } else if (currentPageDiv && (type === 'progress' || type === 'warning')) {
            currentPageDiv.style.display = 'none';
        }

        // Set icon based on type or custom icon (only if element exists)
        if (statusIcon) {
            const icons = {
                info: 'ℹ️',
                warning: '⚠️',
                error: '❌',
                success: '✅',
                progress: '🔄'
            };
            const finalIcon = icon || icons[type] || icons.info;
            statusIcon.textContent = finalIcon;
            void 0;
        }

        // Set message text (remove any manual icons since we use statusIcon)
        if (statusText) {
            // More robust regex to catch all emoji icons with optional whitespace
            const cleanMessage = message.replace(/^[\u2139\u26A0\u274C\u2705\u{1F504}]\uFE0F?\s*/u, '').replace(/^[ℹ️⚠️❌✅🔄]\s*/, '');

            // Convert newlines to <br> tags for HTML display (cart item lists)
            const messageWithBreaks = cleanMessage.replace(/\n/g, '<br>');

            // Include additional info if provided
            let messageHtml = messageWithBreaks;
            if (additionalInfo) {
                messageHtml += `<div class="carbon-status-additional-info">${additionalInfo}</div>`;
            }

            statusText.innerHTML = messageHtml;
            void 0;
        }

        // Handle progress bar (only if element exists)
        if (progressBar) {
            if (showProgress) {
                progressBar.style.display = 'block';
            } else {
                progressBar.style.display = 'none';
            }
        }

        // Clear and setup buttons (only if element exists)
        if (statusButtons) {
            statusButtons.innerHTML = '';
            statusButtons.style.display = 'none';
        }

        // Add existing items display inline if provided
        if (existingItems.length > 0 && statusText) {
            let itemsHtml = '<div class="carbon-status-existing-items">';
            itemsHtml += '<div class="carbon-status-existing-header">Existing Items:</div>';
            existingItems.forEach((item, index) => {
                const productName = item.product_name || 'Unknown Product';
                const carbonFootprint = item.product_carbon_footprint ? `${item.product_carbon_footprint.toFixed(2)} kg CO₂e` : 'Unknown';
                const price = item.price ? `${item.currency || 'USD'} ${item.price}` : 'Unknown';
                const date = item.date_created ? new Date(item.date_created).toLocaleDateString() : 'Unknown';
                const detailsUrl = item.details_url || '#';

                // Match badge disabled for now per design review
                // const matchBadge = item.match_type === 'partial'
                //     ? '<span class="carbon-match-badge partial">PARTIAL MATCH</span>'
                //     : '<span class="carbon-match-badge exact">EXACT MATCH</span>';

                // Item name is now a clickable link to the CarbonGuru footprint page (CG Action Blue)
                // No separate "View details" button - the name itself is the link
                const nameHtml = (detailsUrl && detailsUrl !== '#')
                    ? `<a href="${encodeURI(detailsUrl)}" target="_blank" class="carbon-status-item-name-link">${this.escapeHtml(productName)}</a>`
                    : `<span class="carbon-status-item-name-text">${this.escapeHtml(productName)}</span>`;

                itemsHtml += `
                    <div class="carbon-status-existing-item" data-item-index="${index}">
                        <div class="carbon-status-item-name">${nameHtml}</div>
                        <div class="carbon-status-item-details">
                            <span>${carbonFootprint} | ${price} | ${date}</span>
                        </div>
                    </div>
                `;
            });
            itemsHtml += '</div>';
            statusText.innerHTML += itemsHtml;
            
            // Add click handlers for the existing items
            setTimeout(() => {
                const existingItemElements = statusSection.querySelectorAll('.carbon-status-existing-item');
                existingItemElements.forEach((element, index) => {
                    element.addEventListener('click', async (e) => {
                        // If clicking on the button, handle button click
                        if (e.target.classList.contains('carbon-status-item-button')) {
                            e.stopPropagation();
                            const detailsUrl = e.target.getAttribute('data-details-url');
                            if (detailsUrl && detailsUrl !== '#') {
                                const productUrl = await this.getProductUrl(detailsUrl);
                                window.open(productUrl, '_blank');
                            }
                            return;
                        }
                        
                        // If clicking on the item itself, select it as "use existing"
                        if (this._deduplicationResolve) {
                            const resolve = this._deduplicationResolve;
                            this._deduplicationResolve = null;
                            this.hideStatus();
                            resolve('use_existing');
                        }
                    });
                });
                
                // Add click handlers for the detail buttons
                const detailButtons = statusSection.querySelectorAll('.carbon-status-item-button');
                void 0;
                detailButtons.forEach(button => {
                    const detailsUrl = button.getAttribute('data-details-url');
                    void 0;
                    button.addEventListener('click', async (e) => {
                        e.stopPropagation();
                        e.preventDefault();
                        void 0;
                        if (detailsUrl && detailsUrl !== '#') {
                            void 0;
                            window.open(detailsUrl, '_blank');
                        } else {
                            void 0;
                        }
                    });
                });
            }, 100);
        }

        // Add buttons for interactive status or completion status
        if (buttons.length > 0 && statusButtons) {
            statusButtons.style.display = 'block';
            buttons.forEach((button, index) => {
                const btn = document.createElement('button');
                btn.className = 'carbon-status-button';
                btn.textContent = button.text;
                // Set data-action attribute so buttons can be queried later by handlers
                if (button.action) {
                    btn.setAttribute('data-action', button.action);
                }
                btn.addEventListener('click', async () => {
                    void 0;

                    // Handle deduplication workflow (interactive)
                    if (this._deduplicationResolve) {
                        const resolve = this._deduplicationResolve;
                        this._deduplicationResolve = null;
                        this.hideStatus();
                        resolve(button.action);
                        return;
                    }

                    // Handle completion action buttons
                    if (button.action === 'add_cart_to_basket') {
                        await this.addCartToBasket(button.data);
                    } else if (button.action === 'view_cart_details') {
                        await this.viewCartDetails(button.data);
                    } else if (button.action === 'offset_cart') {
                        await this.offsetCart(button.data);
                    } else if (button.action === 'add_to_basket') {
                        await this.addToBasket(button.data);
                    } else if (button.action === 'view_details') {
                        await this.viewDetails(button.data);
                    } else if (button.action === 'offset_now') {
                        await this.offsetNow(button.data);
                    }
                    // Handle legacy button actions
                    else if (button.action === 'use_existing' && button.data) {
                        this.copyExistingMapping(button.data, window.location.href);
                        this.hideStatus();
                    } else if (button.action === 'create_new') {
                        this.continueWithNewMapping();
                        this.hideStatus();
                    }
                });
                statusButtons.appendChild(btn);
            });
        }

        // Handle timeout countdown
        // FIX: Don't replace innerHTML which destroys click handlers on View Details buttons
        // Instead, append countdown div once and only update its text content
        if (timeout && timeout > 0) {
            let remaining = timeout;

            // Create countdown div once and append it (don't use innerHTML replacement)
            const countdownDiv = document.createElement('div');
            countdownDiv.className = 'carbon-status-countdown';
            countdownDiv.textContent = `Auto-selecting in ${remaining} seconds...`;
            statusText.appendChild(countdownDiv);

            const interval = setInterval(() => {
                remaining--;
                // Only update the countdown text, preserve the rest of the DOM
                countdownDiv.textContent = `Auto-selecting in ${remaining} seconds...`;
                if (remaining <= 0) {
                    clearInterval(interval);
                    if (onTimeout) {
                        onTimeout();
                    }
                    this.hideStatus();
                }
            }, 1000);
            
            statusSection._timeoutInterval = interval;
        }

        // Auto-hide for non-interactive status messages
        if (!interactive && duration > 0) {
            statusSection._statusTimeout = setTimeout(() => {
                this.hideStatus();
            }, duration);
        }

        return statusSection;
    }

    hideStatus() {
        void 0;
        const statusSection = document.getElementById('carbon-status-section');
        if (statusSection) {
            void 0;
            // Clear any timeouts/intervals
            if (statusSection._statusTimeout) {
                clearTimeout(statusSection._statusTimeout);
            }
            if (statusSection._timeoutInterval) {
                clearInterval(statusSection._timeoutInterval);
            }
            
            statusSection.style.display = 'none';
        } else {
            void 0;
        }
    }

    // ========== LEGACY COMPATIBILITY METHOD ==========
    showAnalysisNotification(message, duration = 3000) {
        // Determine notification type based on message content
        let type = 'success';
        if (message.includes('❌') || message.includes('failed') || message.includes('error')) {
            type = 'error';
        } else if (message.includes('⚠️') || message.includes('warning')) {
            type = 'warning';
        } else if (message.includes('🔍') || message.includes('checking') || message.includes('analyzing')) {
            type = 'info';
        }

        this.showStatus(message, { type, duration });
    }

    // REMOVED: fastDuplicateCheck function - was causing processing loops

    // Validate URL to check for existing mappings (via background script cache)
    async validateUrl(url) {
        void 0;
        
        try {
            // Use background script's cached validation
            return new Promise((resolve, reject) => {
                chrome.runtime.sendMessage({
                    action: 'validateUrl',
                    url: url
                }, (response) => {
                    if (chrome.runtime.lastError) {
                        reject(new Error(chrome.runtime.lastError.message));
                    } else {
                        resolve(response);
                    }
                });
            });
        } catch (error) {
            console.error('CarbonGuru: Error validating URL via background script:', error);
            throw error;
        }
    }

    // Legacy direct validation method (fallback)
    async validateUrlDirect(url) {
        void 0;
        
        try {
            const backendUrl = await this.getBackendUrl();
            const response = await fetch(`${backendUrl}/validate_url`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ url: url })
            });

            if (!response.ok) {
                throw new Error(`HTTP error ${response.status}`);
            }

            const data = await response.json();
            void 0;
            
            return {
                success: data.success || false,
                data: data,
                error: data.error,
                has_existing_items: (data.existing_mapped_items && data.existing_mapped_items.length > 0) || false,
                existing_items: data.existing_mapped_items || [],
                has_semantic_similarity: data.has_semantic_similarity || false,
                semantic_similarity: data.semantic_similarity
            };
        } catch (error) {
            console.error('CarbonGuru: Content script error validating URL:', error);
            return {
                success: false,
                error: error.message
            };
        }
    }

    // Show deduplication notification using unified system
    async showDeduplicationNotification(existingItems, url, isCart = false) {
        void 0;
        
        // Prevent duplicate dialogs within timeout period
        const now = Date.now();
        if (this.duplicateDialogShown && (now - this.duplicateDialogTimestamp) < this.duplicateDialogResetTimeout) {
            void 0;
            return 'remap'; // Default to remap to continue processing
        }
        
        // Mark that we're showing a dialog
        this.duplicateDialogShown = true;
        this.duplicateDialogTimestamp = now;
        
        return new Promise((resolve) => {
            // Store resolve function for button handlers with cleanup
            this._deduplicationResolve = (choice) => {
                // Reset duplicate dialog state after choice is made
                setTimeout(() => {
                    this.duplicateDialogShown = false;
                    void 0;
                }, 1000);
                resolve(choice);
            };
            
            const buttons = [
                { text: 'Use Existing', class: 'success', action: 'use_existing' },
                { text: 'New Footprint', class: 'danger', action: 'remap' }
            ];

            const onTimeout = () => {
                this._deduplicationResolve = null;
                resolve('use_existing');
            };

            let message;
            if (isCart) {
                // Enhanced cart message with context
                const firstItem = existingItems[0];
                const cartTitle = firstItem.enhanced_display_title || firstItem.product_name || 'Shopping Cart';
                const itemCount = existingItems.length;
                const totalValue = existingItems.reduce((sum, item) => sum + (parseFloat(item.price) || 0), 0);
                const valueText = totalValue > 0 ? ` worth $${totalValue.toFixed(2)}` : '';
                
                message = `🛒 **${cartTitle}** Found!\n\nPreviously processed cart with ${itemCount} item${itemCount !== 1 ? 's' : ''}${valueText}. Use existing analysis or remap this cart?`;
            } else {
                message = 'Item Recognized: Use existing footprint, or get new footprint?';
            }
            
            this.showStatus(
                message,
                {
                    type: 'warning',
                    interactive: true,
                    buttons: buttons,
                    existingItems: existingItems,
                    timeout: 6,
                    onTimeout: onTimeout
                }
            );
        });
    }

    // Show authentication required prompt
    showAuthenticationPrompt(message, url) {
        void 0;

        const buttons = [
            { text: 'Log In', class: 'success', action: 'login' },
            { text: 'Cancel', class: 'secondary', action: 'cancel' }
        ];

        this.showStatus(
            '🔒 **Authentication Required**\n\nPlease log in to analyze products and view carbon footprints.',
            {
                type: 'warning',
                interactive: true,
                buttons: buttons,
                duration: 0 // Don't auto-dismiss
            }
        );

        // Handle button clicks
        const statusButtons = document.getElementById('carbon-status-buttons');
        if (statusButtons) {
            const loginButton = statusButtons.querySelector('[data-action="login"]');
            const cancelButton = statusButtons.querySelector('[data-action="cancel"]');

            if (loginButton) {
                loginButton.onclick = () => {
                    void 0;
                    // Open popup for authentication
                    browser.runtime.sendMessage({ action: 'openPopup' });
                    this.hideStatus();
                };
            }

            if (cancelButton) {
                cancelButton.onclick = () => {
                    void 0;
                    this.hideStatus();
                };
            }
        }
    }

    hideStatus() {
        const statusSection = document.getElementById('carbon-status-section');
        if (statusSection) {
            statusSection.style.display = 'none';
        }
    }

    // Enhanced deduplication dialog with 10-second timeout
    async showEnhancedDeduplicationDialog(existingItems, url) {
        return new Promise((resolve) => {
            void 0;
            
            const firstItem = existingItems[0];
            const productName = firstItem.product_name || 'Unknown Product';
            const carbonFootprint = firstItem.carbon_footprint ? `${firstItem.carbon_footprint.toFixed(2)} kg CO2e` : 'N/A';
            const dateCreated = firstItem.date_created ? new Date(firstItem.date_created).toLocaleDateString() : 'Unknown';
            
            // Create enhanced dialog overlay
            const overlay = document.createElement('div');
            overlay.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: rgba(0, 0, 0, 0.7);
                z-index: 10000;
                display: flex;
                align-items: center;
                justify-content: center;
                font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            `;
            
            const dialog = document.createElement('div');
            dialog.style.cssText = `
                background: white;
                border-radius: 12px;
                padding: 24px;
                max-width: 500px;
                width: 90%;
                box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
                animation: slideIn 0.3s ease-out;
            `;
            
            // Add animation
            const style = document.createElement('style');
            style.textContent = `
                @keyframes slideIn {
                    from { transform: translateY(-20px); opacity: 0; }
                    to { transform: translateY(0); opacity: 1; }
                }
            `;
            document.head.appendChild(style);
            
            let timeLeft = 10;
            
            dialog.innerHTML = `
                <div style="text-align: center; margin-bottom: 20px;">
                    <h3 style="margin: 0 0 8px 0; color: #2c5aa0; font-size: 20px;">Existing Mapping Found</h3>
                    <p style="margin: 0; color: #666; font-size: 14px;">We found an existing carbon footprint for this product</p>
                </div>
                
                <div style="background: #f8f9fa; border-radius: 8px; padding: 16px; margin-bottom: 20px;">
                    <div style="font-weight: 600; color: #333; margin-bottom: 8px;">${productName}</div>
                    <div style="color: #666; font-size: 14px; margin-bottom: 4px;">Carbon Footprint: <strong>${carbonFootprint}</strong></div>
                    <div style="color: #666; font-size: 14px;">Mapped on: ${dateCreated}</div>
                </div>
                
                <div style="text-align: center; margin-bottom: 20px;">
                    <div style="color: #666; font-size: 14px; margin-bottom: 8px;">
                        Choose an option or wait <span id="countdown" style="font-weight: bold; color: #2c5aa0;">${timeLeft}</span> seconds to use existing data
                    </div>
                </div>
                
                <div style="display: flex; gap: 12px; justify-content: center;">
                    <button id="useExisting" style="
                        background: #2c5aa0;
                        color: white;
                        border: none;
                        padding: 12px 24px;
                        border-radius: 6px;
                        cursor: pointer;
                        font-weight: 600;
                        font-size: 14px;
                        flex: 1;
                        max-width: 140px;
                    ">Use Existing</button>
                    <button id="remap" style="
                        background: #dc3545;
                        color: white;
                        border: none;
                        padding: 12px 24px;
                        border-radius: 6px;
                        cursor: pointer;
                        font-weight: 600;
                        font-size: 14px;
                        flex: 1;
                        max-width: 140px;
                    ">Re-map Product</button>
                </div>
            `;

            overlay.appendChild(dialog);
            document.body.appendChild(overlay);
            
            // Start countdown timer
            const countdownElement = dialog.querySelector('#countdown');
            const countdownInterval = setInterval(() => {
                timeLeft--;
                countdownElement.textContent = timeLeft;
                
                if (timeLeft <= 0) {
                    clearInterval(countdownInterval);
                    document.body.removeChild(overlay);
                    document.head.removeChild(style);
                    void 0;
                    resolve('use_existing'); // Default to existing mapping
                }
            }, 1000);
            
            // Button event handlers
            dialog.querySelector('#useExisting').addEventListener('click', () => {
                clearInterval(countdownInterval);
                document.body.removeChild(overlay);
                document.head.removeChild(style);
                void 0;
                resolve('use_existing');
            });
            
            dialog.querySelector('#remap').addEventListener('click', () => {
                clearInterval(countdownInterval);
                document.body.removeChild(overlay);
                document.head.removeChild(style);
                void 0;
                resolve('remap');
            });
            
            // Close on overlay click
            overlay.addEventListener('click', (e) => {
                if (e.target === overlay) {
                    clearInterval(countdownInterval);
                    document.body.removeChild(overlay);
                    document.head.removeChild(style);
                    void 0;
                    resolve('use_existing'); // Default to existing mapping
                }
            });
        });
    }

    // =============================================================================
    // PHASE 3: Advanced Deduplication Dialog with Confidence Indicators
    // =============================================================================
    
    async showAdvancedDeduplicationDialog(detectionResult, url) {
        /**
         * Phase 3 Enhanced deduplication dialog with:
         * - Confidence indicators (High/Medium/Low)
         * - Match evidence display (UPC matched, SKU matched, etc.)
         * - Smart recommendations based on confidence
         * - Multiple match handling with comparison view
         * 
         * @param {Object} detectionResult - Phase 3 detection result
         * @param {string} url - Current page URL
         * @returns {Promise<string>} User choice: 'use_existing', 'create_new', etc.
         */
        return new Promise((resolve) => {
            void 0;
            
            const matches = detectionResult.matches || [];
            const confidence = detectionResult.total_confidence || detectionResult.confidence_score || 0;
            const matchMethod = detectionResult.primary_method || detectionResult.match_method || 'unknown';
            const evidence = detectionResult.evidence || {};
            
            if (!matches.length) {
                void 0;
                resolve('create_new');
                return;
            }
            
            // Determine confidence level and styling
            const confidenceLevel = this._getConfidenceLevel(confidence);
            const confidenceConfig = this._getConfidenceConfig(confidenceLevel);
            
            // Create enhanced dialog overlay
            const overlay = document.createElement('div');
            overlay.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: rgba(0, 0, 0, 0.8);
                z-index: 10001;
                display: flex;
                align-items: center;
                justify-content: center;
                font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
                backdrop-filter: blur(4px);
            `;
            
            // Create dialog content
            const dialog = document.createElement('div');
            dialog.style.cssText = `
                background: white;
                border-radius: 12px;
                box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
                max-width: 600px;
                width: 90%;
                max-height: 80vh;
                overflow-y: auto;
                position: relative;
            `;
            
            // Build dialog HTML
            dialog.innerHTML = this._buildAdvancedDialogHTML(matches, confidence, confidenceConfig, evidence, matchMethod);
            
            // Add CSS styles
            const style = document.createElement('style');
            style.textContent = this._getAdvancedDialogCSS();
            document.head.appendChild(style);
            
            overlay.appendChild(dialog);
            document.body.appendChild(overlay);
            
            // Setup event handlers
            this._setupAdvancedDialogHandlers(overlay, dialog, matches, resolve);
            
            // Auto-close timer (30 seconds for complex decisions)
            const autoCloseTimer = setTimeout(() => {
                if (document.body.contains(overlay)) {
                    this._closeAdvancedDialog(overlay, style);
                    void 0;
                    resolve('create_new');
                }
            }, 30000);
            
            // Store timer for cleanup
            overlay._autoCloseTimer = autoCloseTimer;
        });
    }
    
    _getConfidenceLevel(confidence) {
        if (confidence >= 90) return 'high';
        if (confidence >= 80) return 'medium';
        return 'low';
    }
    
    _getConfidenceConfig(level) {
        const configs = {
            'high': {
                icon: '🎯',
                color: '#28a745',
                bgColor: '#d4edda',
                borderColor: '#c3e6cb',
                text: 'High Confidence Match',
                recommendation: 'Recommended: Use existing calculation',
                description: 'Strong identifier match with high confidence'
            },
            'medium': {
                icon: '⚠️',
                color: '#ffc107',
                bgColor: '#fff3cd',
                borderColor: '#ffeaa7',
                text: 'Medium Confidence',
                recommendation: 'Likely match - review details',
                description: 'Good match but please verify the details'
            },
            'low': {
                icon: '❓',
                color: '#dc3545',
                bgColor: '#f8d7da',
                borderColor: '#f5c6cb',
                text: 'Low Confidence',
                recommendation: 'Possible match - verify manually',
                description: 'Uncertain match - manual verification recommended'
            }
        };
        return configs[level] || configs['low'];
    }
    
    _buildAdvancedDialogHTML(matches, confidence, config, evidence, matchMethod) {
        const primaryMatch = matches[0];
        const hasMultipleMatches = matches.length > 1;
        
        // Build evidence display
        const evidenceHTML = this._buildEvidenceHTML(evidence, primaryMatch);
        
        // Build matches display
        const matchesHTML = hasMultipleMatches 
            ? this._buildMultipleMatchesHTML(matches)
            : this._buildSingleMatchHTML(primaryMatch);
        
        return `
            <div class="carbon-advanced-dialog">
                <div class="carbon-dialog-header">
                    <h3>
                        ${config.icon} Duplicate Product Detected
                    </h3>
                    <button class="carbon-close-btn" data-action="create_new">&times;</button>
                </div>
                
                <div class="carbon-confidence-indicator" style="
                    background: ${config.bgColor}; 
                    border: 1px solid ${config.borderColor}; 
                    color: ${config.color};
                ">
                    <div class="carbon-confidence-header">
                        <span class="carbon-confidence-text">${config.text} (${confidence.toFixed(1)}%)</span>
                        <span class="carbon-match-method">via ${matchMethod.replace('_', ' ')}</span>
                    </div>
                    <div class="carbon-confidence-desc">${config.description}</div>
                </div>
                
                ${evidenceHTML}
                
                <div class="carbon-matches-section">
                    <h4>${hasMultipleMatches ? `${matches.length} Matches Found` : 'Match Details'}</h4>
                    ${matchesHTML}
                </div>
                
                <div class="carbon-recommendation">
                    <strong>Recommendation:</strong> ${config.recommendation}
                </div>
                
                <div class="carbon-dialog-actions">
                    ${this._buildActionButtons(confidence, hasMultipleMatches)}
                </div>
            </div>
        `;
    }
    
    _buildEvidenceHTML(evidence, match) {
        const identifiersMatched = evidence.identifiers_matched || [];
        const exactMatches = evidence.exact_matches || [];
        const partialMatches = evidence.partial_matches || [];
        
        if (!identifiersMatched.length) {
            return '<div class="carbon-evidence-section"><em>No identifier evidence available</em></div>';
        }
        
        const evidenceItems = identifiersMatched.map(identifier => {
            const isExact = exactMatches.includes(identifier);
            const isPartial = partialMatches.includes(identifier);
            const icon = isExact ? '✅' : (isPartial ? '⚠️' : '❓');
            const type = isExact ? 'Exact Match' : (isPartial ? 'Partial Match' : 'Match');
            const value = match.item_data?.identifiers?.[identifier] || 'N/A';
            
            return `
                <div class="carbon-evidence-item">
                    ${icon} <strong>${identifier.toUpperCase()}:</strong> ${value} <em>(${type})</em>
                </div>
            `;
        }).join('');
        
        return `
            <div class="carbon-evidence-section">
                <h4>Match Evidence:</h4>
                ${evidenceItems}
            </div>
        `;
    }
    
    _buildSingleMatchHTML(match) {
        const product = match.item_data || {};
        const carbonFootprint = product.carbon_footprint || product.product_carbon_footprint;
        const confidence = match.confidence || 0;
        
        return `
            <div class="carbon-match-item single-match">
                <div class="carbon-match-header">
                    <h5>${product.product_name || 'Unknown Product'}</h5>
                    <span class="carbon-match-confidence">${confidence.toFixed(1)}% confidence</span>
                </div>
                <div class="carbon-match-details">
                    <div class="carbon-detail-row">
                        <span class="carbon-label">Carbon Footprint:</span>
                        <span class="carbon-value">${carbonFootprint ? carbonFootprint.toFixed(2) + ' kg CO₂e' : 'N/A'}</span>
                    </div>
                    <div class="carbon-detail-row">
                        <span class="carbon-label">Previously analyzed:</span>
                        <span class="carbon-value">${product.date_created ? new Date(product.date_created).toLocaleDateString() : 'Unknown'}</span>
                    </div>
                </div>
            </div>
        `;
    }
    
    _buildMultipleMatchesHTML(matches) {
        return `
            <div class="carbon-matches-grid">
                ${matches.map((match, index) => {
                    const product = match.item_data || {};
                    const carbonFootprint = product.carbon_footprint || product.product_carbon_footprint;
                    const confidence = match.confidence || 0;
                    
                    return `
                        <div class="carbon-match-item" data-match-index="${index}">
                            <div class="carbon-match-header">
                                <h6>${product.product_name || `Match ${index + 1}`}</h6>
                                <span class="carbon-match-confidence">${confidence.toFixed(1)}%</span>
                            </div>
                            <div class="carbon-match-summary">
                                <div>Carbon: ${carbonFootprint ? carbonFootprint.toFixed(2) + ' kg CO₂e' : 'N/A'}</div>
                                <div>Date: ${product.date_created ? new Date(product.date_created).toLocaleDateString() : 'Unknown'}</div>
                            </div>
                            <button class="carbon-select-match-btn" data-action="use_match" data-match-index="${index}">
                                Select This Match
                            </button>
                        </div>
                    `;
                }).join('')}
            </div>
        `;
    }
    
    _buildActionButtons(confidence, hasMultipleMatches) {
        if (hasMultipleMatches) {
            return `
                <button class="carbon-btn carbon-btn-primary" data-action="use_best_match">
                    Use Best Match
                </button>
                <button class="carbon-btn carbon-btn-secondary" data-action="review_all">
                    Review All Matches
                </button>
                <button class="carbon-btn carbon-btn-outline" data-action="create_new">
                    Create New Analysis
                </button>
            `;
        } else if (confidence >= 90) {
            return `
                <button class="carbon-btn carbon-btn-success" data-action="use_existing">
                    Use This Match
                </button>
                <button class="carbon-btn carbon-btn-outline" data-action="create_new">
                    Create New
                </button>
            `;
        } else if (confidence >= 80) {
            return `
                <button class="carbon-btn carbon-btn-warning" data-action="use_existing">
                    Use This Match
                </button>
                <button class="carbon-btn carbon-btn-primary" data-action="create_new">
                    Create New Analysis
                </button>
                <button class="carbon-btn carbon-btn-link" data-action="get_help">
                    Need Help?
                </button>
            `;
        } else {
            return `
                <button class="carbon-btn carbon-btn-primary" data-action="create_new">
                    Create New Analysis
                </button>
                <button class="carbon-btn carbon-btn-secondary" data-action="use_existing">
                    Use Anyway
                </button>
                <button class="carbon-btn carbon-btn-link" data-action="get_help">
                    Get Help
                </button>
            `;
        }
    }
    
    _getAdvancedDialogCSS() {
        return `
            .carbon-advanced-dialog {
                padding: 24px;
                font-size: 14px;
                line-height: 1.5;
            }
            
            .carbon-dialog-header {
                display: flex;
                justify-content: space-between;
                align-items: center;
                margin-bottom: 20px;
                padding-bottom: 12px;
                border-bottom: 1px solid #eee;
            }
            
            .carbon-dialog-header h3 {
                margin: 0;
                font-size: 18px;
                font-weight: 600;
                color: #333;
            }
            
            .carbon-close-btn {
                background: none;
                border: none;
                font-size: 24px;
                cursor: pointer;
                color: #999;
                padding: 0;
                width: 30px;
                height: 30px;
                display: flex;
                align-items: center;
                justify-content: center;
                border-radius: 50%;
                transition: all 0.2s;
            }
            
            .carbon-close-btn:hover {
                background: #f5f5f5;
                color: #333;
            }
            
            .carbon-confidence-indicator {
                padding: 16px;
                border-radius: 8px;
                margin-bottom: 20px;
            }
            
            .carbon-confidence-header {
                display: flex;
                justify-content: space-between;
                align-items: center;
                margin-bottom: 8px;
            }
            
            .carbon-confidence-text {
                font-weight: 600;
                font-size: 16px;
            }
            
            .carbon-match-method {
                font-size: 12px;
                opacity: 0.8;
                text-transform: capitalize;
            }
            
            .carbon-confidence-desc {
                font-size: 13px;
                opacity: 0.9;
            }
            
            .carbon-evidence-section {
                margin-bottom: 20px;
            }
            
            .carbon-evidence-section h4 {
                margin: 0 0 12px 0;
                font-size: 14px;
                font-weight: 600;
                color: #555;
            }
            
            .carbon-evidence-item {
                padding: 8px 0;
                border-bottom: 1px solid #f0f0f0;
                font-size: 13px;
            }
            
            .carbon-evidence-item:last-child {
                border-bottom: none;
            }
            
            .carbon-matches-section {
                margin-bottom: 20px;
            }
            
            .carbon-matches-section h4 {
                margin: 0 0 16px 0;
                font-size: 14px;
                font-weight: 600;
                color: #555;
            }
            
            .carbon-match-item {
                border: 1px solid #e0e0e0;
                border-radius: 8px;
                padding: 16px;
                margin-bottom: 12px;
                background: #fafafa;
            }
            
            .carbon-match-item.single-match {
                background: white;
                border-color: #d0d0d0;
            }
            
            .carbon-match-header {
                display: flex;
                justify-content: space-between;
                align-items: center;
                margin-bottom: 12px;
            }
            
            .carbon-match-header h5,
            .carbon-match-header h6 {
                margin: 0;
                font-size: 15px;
                font-weight: 600;
                color: #333;
            }
            
            .carbon-match-confidence {
                background: #e9ecef;
                color: #495057;
                padding: 4px 8px;
                border-radius: 12px;
                font-size: 12px;
                font-weight: 500;
            }
            
            .carbon-detail-row {
                display: flex;
                justify-content: space-between;
                margin-bottom: 8px;
            }
            
            .carbon-detail-row:last-child {
                margin-bottom: 0;
            }
            
            .carbon-label {
                color: #666;
                font-size: 13px;
            }
            
            .carbon-value {
                font-weight: 500;
                font-size: 13px;
                color: #333;
            }
            
            .carbon-matches-grid {
                display: grid;
                gap: 12px;
            }
            
            .carbon-match-summary {
                font-size: 12px;
                color: #666;
                margin-bottom: 12px;
            }
            
            .carbon-match-summary div {
                margin-bottom: 4px;
            }
            
            .carbon-select-match-btn {
                background: #007bff;
                color: white;
                border: none;
                padding: 8px 16px;
                border-radius: 6px;
                font-size: 12px;
                cursor: pointer;
                width: 100%;
                transition: background 0.2s;
            }
            
            .carbon-select-match-btn:hover {
                background: #0056b3;
            }
            
            .carbon-recommendation {
                background: #f8f9fa;
                border: 1px solid #e9ecef;
                border-radius: 6px;
                padding: 12px;
                margin-bottom: 20px;
                font-size: 13px;
                color: #495057;
            }
            
            .carbon-dialog-actions {
                display: flex;
                gap: 12px;
                justify-content: flex-end;
                flex-wrap: wrap;
            }
            
            .carbon-btn {
                padding: 10px 20px;
                border-radius: 6px;
                border: none;
                font-size: 14px;
                font-weight: 500;
                cursor: pointer;
                transition: all 0.2s;
                text-decoration: none;
                display: inline-flex;
                align-items: center;
                justify-content: center;
                min-width: 100px;
            }
            
            .carbon-btn-primary {
                background: #007bff;
                color: white;
            }
            
            .carbon-btn-primary:hover {
                background: #0056b3;
            }
            
            .carbon-btn-success {
                background: #28a745;
                color: white;
            }
            
            .carbon-btn-success:hover {
                background: #1e7e34;
            }
            
            .carbon-btn-warning {
                background: #ffc107;
                color: #212529;
            }
            
            .carbon-btn-warning:hover {
                background: #e0a800;
            }
            
            .carbon-btn-secondary {
                background: #6c757d;
                color: white;
            }
            
            .carbon-btn-secondary:hover {
                background: #545b62;
            }
            
            .carbon-btn-outline {
                background: white;
                color: #007bff;
                border: 1px solid #007bff;
            }
            
            .carbon-btn-outline:hover {
                background: #007bff;
                color: white;
            }
            
            .carbon-btn-link {
                background: none;
                color: #007bff;
                text-decoration: underline;
                border: none;
                min-width: auto;
                padding: 10px;
            }
            
            .carbon-btn-link:hover {
                color: #0056b3;
            }
            
            @media (max-width: 600px) {
                .carbon-dialog-actions {
                    flex-direction: column;
                }
                
                .carbon-btn {
                    width: 100%;
                }
                
                .carbon-match-header {
                    flex-direction: column;
                    align-items: flex-start;
                    gap: 8px;
                }
            }
        `;
    }
    
    _setupAdvancedDialogHandlers(overlay, dialog, matches, resolve) {
        // Handle button clicks
        dialog.addEventListener('click', (e) => {
            const action = e.target.dataset.action;
            const matchIndex = e.target.dataset.matchIndex;
            
            if (action) {
                clearTimeout(overlay._autoCloseTimer);
                
                if (action === 'use_match' || action === 'use_best_match') {
                    const selectedMatch = action === 'use_best_match' ? matches[0] : matches[parseInt(matchIndex) || 0];
                    this._closeAdvancedDialog(overlay);
                    void 0;
                    resolve('use_existing');
                } else if (action === 'use_existing') {
                    this._closeAdvancedDialog(overlay);
                    void 0;
                    resolve('use_existing');
                } else if (action === 'create_new') {
                    this._closeAdvancedDialog(overlay);
                    void 0;
                    resolve('create_new');
                } else if (action === 'review_all') {
                    // Expand all matches view
                    void 0;
                    // For now, just highlight the matches section
                    const matchesSection = dialog.querySelector('.carbon-matches-section');
                    if (matchesSection) {
                        matchesSection.scrollIntoView({ behavior: 'smooth' });
                        matchesSection.style.border = '2px solid #007bff';
                        setTimeout(() => {
                            matchesSection.style.border = '';
                        }, 2000);
                    }
                } else if (action === 'get_help') {
                    this._closeAdvancedDialog(overlay);
                    void 0;
                    resolve('get_help');
                }
            }
        });
        
        // Close on overlay click
        overlay.addEventListener('click', (e) => {
            if (e.target === overlay) {
                clearTimeout(overlay._autoCloseTimer);
                this._closeAdvancedDialog(overlay);
                void 0;
                resolve('create_new');
            }
        });
        
        // Close on Escape key
        const handleEscape = (e) => {
            if (e.key === 'Escape') {
                clearTimeout(overlay._autoCloseTimer);
                this._closeAdvancedDialog(overlay);
                document.removeEventListener('keydown', handleEscape);
                void 0;
                resolve('create_new');
            }
        };
        document.addEventListener('keydown', handleEscape);
    }
    
    _closeAdvancedDialog(overlay, style) {
        if (overlay._autoCloseTimer) {
            clearTimeout(overlay._autoCloseTimer);
        }
        
        if (document.body.contains(overlay)) {
            document.body.removeChild(overlay);
        }
        
        // Remove styles
        const existingStyles = document.querySelectorAll('style');
        existingStyles.forEach(styleEl => {
            if (styleEl.textContent.includes('.carbon-advanced-dialog')) {
                document.head.removeChild(styleEl);
            }
        });
    }

    // Copy existing mapping from content script via background script
    // Routes through background to avoid CSP/CORS issues in MV3 content scripts
    async copyExistingMapping(existingItem, url) {
        void 0;

        if (!existingItem || !existingItem.id || !existingItem.session_id) {
            this.showStatus('Error copying mapping: missing item data', {
                type: 'error',
                duration: 4000
            });
            return;
        }

        try {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const data = await new Promise((resolve, reject) => {
                runtime.sendMessage({
                    action: 'copyMappingToSession',
                    copyData: {
                        source_item_id: existingItem.id,
                        source_session_id: existingItem.session_id,
                        current_session_id: 'extension_content_session',
                        url: url || window.location.href
                    }
                }, (response) => {
                    if (runtime.lastError) {
                        reject(new Error(runtime.lastError.message));
                    } else {
                        resolve(response);
                    }
                });
            });

            if (data && data.success) {
                void 0;
                void 0;

                // Display the carbon footprint results immediately
                // Use the copy_existing_mapping response data which includes PCF comparison and social cost
                this.completeProgress();

                const resultsData = {
                    product_carbon_footprint: data.carbon_footprint,
                    social_cost_usd: data.social_cost_usd,
                    // Offset cost fields for immediate display
                    offset_cost_usd: data.offset_cost_usd,
                    display_cost_usd: data.display_cost_usd,
                    orderable_cost_usd: data.orderable_cost_usd,
                    pcf_comparison: data.pcf_comparison,
                    naics_code: data.naics_code,
                    naics_title: data.naics_title,
                    lca_type: data.lca_type,
                    product_name: data.product_name,
                    details_url: data.redirect_url,
                    carbonguru_id: data.carbonguru_id  // Needed for WooCommerce URL polling
                };

                void 0;
                await this.showResults(resultsData);

                // Notify popup to refresh
                runtime.sendMessage({
                    action: 'mappingCopied',
                    data: data
                });
            } else {
                throw new Error((data && data.error) || 'Failed to copy mapping');
            }
        } catch (error) {
            console.error('CarbonGuru: Content script error copying existing mapping:', error);
            this.showStatus(`Error copying mapping: ${error.message}`, {
                type: 'error',
                duration: 4000
            });
        }
    }

    // ========== EXPANDABLE TILE FUNCTIONALITY ==========
    
    async startAnalysis(skipAuthCheck = false) {
        void 0;

        // Immediately expand the tile so the user sees visual feedback
        this.expandTileWithoutAuthCheck();

        // Check authentication first (unless explicitly skipped)
        if (!skipAuthCheck) {
            const authStatus = await this.checkAuthentication();
            if (!authStatus.isAuthenticated) {
                void 0;
                this.showAuthenticationPrompt();
                return;
            }
        } else {
            void 0;
        }

        // GMAIL PAGE HANDLING
        if (this.isGmailPage && this.gmailDetector && this.gmailIntegration) {
            void 0;

            // Check page type
            const pageType = this.gmailDetector.pageType;

            if (pageType === 'email_view') {
                // Single email view - process directly
                await this.processGmailSingleEmail();
            } else if (pageType === 'inbox_list' || pageType === 'category_view') {
                // Inbox/list view OR category view (e.g., #category/purchases) - show scope selector
                await this.processGmailBatchEmails();
            } else {
                // Unknown page type
                this.showStatus('Gmail page type not recognized', { type: 'warning', duration: 3000 });
            }

            return; // Exit early - Gmail processing handled
        }

        // NORMAL PAGE HANDLING (non-Gmail)
        // Show progress section
        this.showProgress();

        // Hide results and notifications
        this.hideResults();

        // Start the actual analysis
        this.extractFullPage();
    }

    async waitForBackgroundInit(maxAttempts = 20) {
        // Wait up to 10 seconds for background script to initialize
        void 0;

        for (let attempt = 1; attempt <= maxAttempts; attempt++) {
            try {
                const response = await browser.runtime.sendMessage({ action: 'ping' });

                if (response && response.pong && response.initialized) {
                    void 0;
                    return true;
                }

                void 0;
            } catch (error) {
                void 0;
            }

            // Wait 500ms before next attempt
            await new Promise(resolve => setTimeout(resolve, 500));
        }

        void 0;
        return false;
    }

    async checkAuthentication(bustCache = false) {
        // Wait for background script to be ready
        const isBackgroundReady = await this.waitForBackgroundInit();

        if (!isBackgroundReady) {
            console.error('CarbonGuru: Background script not ready - authentication check failed');
            return { isAuthenticated: false, reason: 'Background script not initialized' };
        }

        return new Promise((resolve) => {
            const message = {
                action: 'checkAuthStatus'
            };

            // Add cache-busting parameter if requested
            if (bustCache) {
                message.bustCache = true;
                void 0;
            }

            browser.runtime.sendMessage(message, (response) => {
                void 0;

                if (!response) {
                    console.error('CarbonGuru: No response from background script');
                    resolve({ isAuthenticated: false, reason: 'No response from background' });
                    return;
                }

                // Return the full response including reason field
                void 0;
                resolve(response);
            });
        });
    }

    showAuthenticationPrompt() {
        // Keep service worker alive while user decides to authenticate
        if (window.cgKeepalive) window.cgKeepalive.start();

        // Show a user-friendly authentication prompt in the tile
        // Use expandTileWithoutAuthCheck to avoid a second async auth round-trip
        // (startAnalysis already checked auth before calling this)
        this.expandTileWithoutAuthCheck();

        const statusSection = document.getElementById('carbon-status-section');
        if (!statusSection) return;

        statusSection.style.display = 'block';
        statusSection.innerHTML = `
            <div class="carbon-status-message" style="padding: 20px;">
                <div style="font-size: 16px; font-weight: bold; margin-bottom: 10px;">
                    🔐 Authentication Required
                </div>
                <div style="margin-bottom: 15px;">
                    Please sign in with your WordPress account to analyze products.
                </div>
                <button id="carbon-auth-prompt-btn" style="
                    background: #007bff;
                    color: white;
                    border: none;
                    padding: 10px 20px;
                    border-radius: 4px;
                    cursor: pointer;
                    font-size: 14px;
                    font-weight: bold;
                ">
                    Sign in with WordPress
                </button>
                <div id="carbon-auth-status" style="margin-top: 10px; font-size: 12px; color: #666;"></div>
            </div>
        `;

        // Add click handler for auth button
        const authBtn = document.getElementById('carbon-auth-prompt-btn');
        const authStatus = document.getElementById('carbon-auth-status');

        if (authBtn) {
            authBtn.addEventListener('click', async () => {
                void 0;

                // Disable button and show status
                authBtn.disabled = true;
                if (authStatus) authStatus.textContent = 'Opening authorization page...';

                try {
                    // Trigger OAuth flow via background script
                    const response = await browser.runtime.sendMessage({action: 'initiateOAuth'});

                    if (response && response.success) {
                        if (authStatus) authStatus.textContent = 'Please complete authorization in the opened tab...';
                        void 0;

                        // After successful OAuth initiation, we could listen for auth completion
                        // and automatically retry the analysis, but for now just show success
                    } else if (!response) {
                        // No response = service worker likely not ready (shim returned undefined)
                        void 0;
                        if (authStatus) {
                            authStatus.innerHTML = `
                                <span style="color: #d32f2f;">❌ Extension reloading...</span><br>
                                <span style="color: #666; font-size: 11px;">Click the button again to retry, or refresh the page.</span>
                            `;
                        }
                        authBtn.disabled = false;
                        authBtn.textContent = 'Try Again';
                        authBtn.style.background = '#ff9800';
                    } else {
                        const errorMsg = response?.error || 'Failed to initiate OAuth';
                        console.error('CarbonGuru: OAuth initiation failed:', errorMsg);
                        if (authStatus) {
                            authStatus.textContent = `❌ ${errorMsg}`;
                            authStatus.style.color = '#d32f2f';
                        }
                        authBtn.disabled = false;
                    }
                } catch (error) {
                    console.error('CarbonGuru: OAuth initiation error:', error);
                    const errorStr = String(error.message || error || '').toLowerCase();
                    const isServiceWorkerError = errorStr.includes('context invalidated') ||
                        errorStr.includes('service worker') ||
                        errorStr.includes('disconnected') ||
                        errorStr.includes('receiving end does not exist');

                    if (authStatus) {
                        if (isServiceWorkerError) {
                            // Show user-friendly message with retry guidance
                            authStatus.innerHTML = `
                                <span style="color: #d32f2f;">❌ Extension reloading...</span><br>
                                <span style="color: #666; font-size: 11px;">Click the button again to retry, or refresh the page.</span>
                            `;
                        } else {
                            authStatus.textContent = `❌ ${error.message}`;
                            authStatus.style.color = '#d32f2f';
                        }
                    }
                    authBtn.disabled = false;

                    // For service worker errors, also reset the button text to encourage retry
                    if (isServiceWorkerError) {
                        authBtn.textContent = 'Try Again';
                        authBtn.style.background = '#ff9800'; // Orange to indicate retry
                    }
                }
            });
        }
    }

    handleAuthenticationChange(isAuthenticated, user) {
        void 0;

        if (isAuthenticated) {
            // User just logged in - show success message and hide auth prompt
            const statusSection = document.getElementById('carbon-status-section');
            if (statusSection) {
                statusSection.style.display = 'block';
                statusSection.innerHTML = `
                    <div class="carbon-status-message" style="padding: 20px; background: #e8f5e9;">
                        <div style="font-size: 16px; font-weight: bold; margin-bottom: 10px; color: #2e7d32;">
                            ✅ Authentication Successful
                        </div>
                        <div style="margin-bottom: 15px; color: #1b5e20;">
                            Signed in as ${user ? user.email : 'user'}. You can now analyze products!
                        </div>
                        <button id="carbon-auth-retry-btn" style="
                            background: #4caf50;
                            color: white;
                            border: none;
                            padding: 10px 20px;
                            border-radius: 4px;
                            cursor: pointer;
                            font-size: 14px;
                            font-weight: bold;
                        ">
                            Get Carbon Footprint Now
                        </button>
                    </div>
                `;

                // Add click handler to retry analysis
                const retryBtn = document.getElementById('carbon-auth-retry-btn');
                if (retryBtn) {
                    retryBtn.addEventListener('click', () => {
                        void 0;
                        this.startAnalysis();
                    });
                }

                // Auto-hide success message after 5 seconds
                setTimeout(() => {
                    if (statusSection) {
                        statusSection.style.display = 'none';
                        statusSection.innerHTML = '';
                    }
                }, 5000);
            }

            // Load recent items now that the user is authenticated
            this.loadRecentItems();
        } else {
            // User logged out - clear any auth-related UI
            const statusSection = document.getElementById('carbon-status-section');
            if (statusSection) {
                statusSection.style.display = 'none';
                statusSection.innerHTML = '';
            }
        }
    }
    
    setupTileToggleListener() {
        // Use event delegation on the tile container to ensure it works
        const tile = document.getElementById('carbon-guru-tile');
        if (tile) {
            tile.addEventListener('click', (e) => {
                // Check if click is on toggle button or its contents
                const toggleBtn = e.target.closest('#carbon-tile-toggle');
                if (toggleBtn) {
                    e.stopPropagation();
                    e.preventDefault();
                    void 0;
                    this.toggleTile();
                }
            });
        }
    }
    
    setupFloatingTileBasketListener() {
        const basketBtn = document.getElementById('carbon-add-to-basket-btn');
        if (basketBtn) {
            basketBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.addCurrentItemToBasket();
            });
        }
    }
    
    addCurrentItemToBasket() {
        // Get the current analyzed item data
        if (this.currentAnalysisResult) {
            const item = {
                id: this.currentAnalysisResult.id || `floating_${Date.now()}`,
                product_name: this.currentAnalysisResult.product_name || this.currentAnalysisResult.name || 'Current Product',
                product_carbon_footprint: this.currentAnalysisResult.product_carbon_footprint || this.currentAnalysisResult.carbon_footprint,
                co2e_per_dollar: this.currentAnalysisResult.co2e_per_dollar,
                details_url: this.currentAnalysisResult.details_url,
                date_created: new Date().toISOString(),
                source: 'floating_tile'
            };
            
            // Send to background script
            if (typeof browser !== 'undefined' && browser.runtime) {
                browser.runtime.sendMessage({
                    action: 'addToBasket',
                    item: item
                }).then(response => {
                    if (response && response.success) {
                        // Show success feedback
                        const btn = document.getElementById('carbon-add-to-basket-btn');
                        if (btn) {
                            const originalText = btn.textContent;
                            btn.textContent = 'Added!';
                            btn.style.background = '#28a745';
                            setTimeout(() => {
                                btn.textContent = originalText;
                                btn.style.background = '';
                            }, 2000);
                        }
                    } else {
                        console.error('CarbonGuru: Failed to add item to basket from floating tile');
                    }
                }).catch(error => {
                    console.error('CarbonGuru: Error adding item to basket from floating tile:', error);
                });
            }
        } else {
            void 0;
        }
    }
    
    toggleTile() {
        const content = document.getElementById('carbon-tile-content');
        const toggle = document.getElementById('carbon-tile-toggle');
        
        if (content && toggle) {
            const isExpanded = content.style.display === 'block';
            
            if (isExpanded) {
                this.collapseTile();
            } else {
                this.expandTile();
            }
        }
    }
    
    async expandTile() {
        // Clear minimal state timer and remove minimal classes (tile panel is opening)
        this.clearMinimalStateTimer();
        const minBtn = document.getElementById('carbon-analyze-btn');
        if (minBtn) minBtn.classList.remove('cg-minimal');
        const minHeader = document.getElementById('carbon-tile-header');
        if (minHeader) minHeader.classList.remove('cg-header-minimal');
        const minTile = document.getElementById('carbon-guru-tile');
        if (minTile) minTile.classList.remove('cg-tile-minimal');

        const content = document.getElementById('carbon-tile-content');
        const toggle = document.getElementById('carbon-tile-toggle');

        if (content && toggle) {
            content.style.display = 'block';
            toggle.style.display = 'inline-block';
            toggle.textContent = '▲';
            toggle.title = 'Collapse';
            toggle.setAttribute('aria-expanded', 'true');

            // Save tile state to persist across page refreshes (extension-private storage)
            browser.storage.local.set({ carbonguru_tile_expanded: true });

            // Show current page when tile is expanded
            this.showCurrentPage();

            // Proactively check authentication status when tile is shown
            // This ensures the floating tile reflects the current auth state
            void 0;
            const authStatus = await this.checkAuthentication(true); // Force cache bust

            if (authStatus.isAuthenticated) {
                void 0;
                this.updateAuthenticatedUI(authStatus.user);
            } else {
                void 0;
                // UI will show "Analyze" button which will trigger auth check
            }
        }
    }

    /**
     * Expand the tile UI without triggering an async auth check.
     * Used by startAnalysis() and showAuthenticationPrompt() to provide
     * immediate visual feedback before the auth round-trip completes.
     */
    expandTileWithoutAuthCheck() {
        // Clear minimal state timer and remove minimal classes (tile panel is opening)
        this.clearMinimalStateTimer();
        const minBtn = document.getElementById('carbon-analyze-btn');
        if (minBtn) minBtn.classList.remove('cg-minimal');
        const minHeader = document.getElementById('carbon-tile-header');
        if (minHeader) minHeader.classList.remove('cg-header-minimal');
        const minTile = document.getElementById('carbon-guru-tile');
        if (minTile) minTile.classList.remove('cg-tile-minimal');

        const content = document.getElementById('carbon-tile-content');
        const toggle = document.getElementById('carbon-tile-toggle');

        if (content && toggle) {
            content.style.display = 'block';
            toggle.style.display = 'inline-block';
            toggle.textContent = '▲';
            toggle.title = 'Collapse';
            toggle.setAttribute('aria-expanded', 'true');

            // Save tile state to persist across page refreshes (extension-private storage)
            browser.storage.local.set({ carbonguru_tile_expanded: true });

            // Show current page when tile is expanded
            this.showCurrentPage();
        }
    }

    updateAuthenticatedUI(user) {
        // Update UI elements to reflect authenticated state
        void 0;

        // Hide authentication prompt if it's currently showing
        const statusSection = document.getElementById('carbon-status-section');
        if (statusSection && statusSection.innerHTML.includes('Authentication Required')) {
            statusSection.style.display = 'none';
        }

        // Could add user info display here in the future
        // For now, the tile will function normally for authenticated users
    }

    collapseTile() {
        const content = document.getElementById('carbon-tile-content');
        const toggle = document.getElementById('carbon-tile-toggle');

        if (content && toggle) {
            content.style.display = 'none';
            toggle.textContent = '▼';
            toggle.title = 'Expand';
            toggle.setAttribute('aria-expanded', 'false');

            // Save tile state to persist across page refreshes (extension-private storage)
            browser.storage.local.set({ carbonguru_tile_expanded: false });
        }

        // Restart minimal-state timer when tile panel is collapsed
        this.startMinimalStateTimer();
    }

    // ---- Minimal state auto-collapse ----

    startMinimalStateTimer() {
        this.clearMinimalStateTimer();

        // Don't start timer if analyzing or tile content is expanded
        const content = document.getElementById('carbon-tile-content');
        if (this.isAnalyzing || (content && content.style.display === 'block')) {
            return;
        }

        this._minimalStateTimerId = setTimeout(() => {
            this.collapseToMinimal();
        }, 120000); // 120 seconds
    }

    clearMinimalStateTimer() {
        if (this._minimalStateTimerId) {
            clearTimeout(this._minimalStateTimerId);
            this._minimalStateTimerId = null;
        }
    }

    collapseToMinimal() {
        const btn = document.getElementById('carbon-analyze-btn');
        const header = document.getElementById('carbon-tile-header');
        const tile = document.getElementById('carbon-guru-tile');
        const content = document.getElementById('carbon-tile-content');

        // Don't collapse if analyzing or tile is expanded
        if (this.isAnalyzing || (content && content.style.display === 'block')) {
            return;
        }

        if (btn) {
            btn.classList.add('cg-minimal');
        }
        if (header) {
            header.classList.add('cg-header-minimal');
        }
        if (tile) {
            tile.classList.add('cg-tile-minimal');
        }
    }

    expandFromMinimal() {
        const btn = document.getElementById('carbon-analyze-btn');
        const header = document.getElementById('carbon-tile-header');
        const tile = document.getElementById('carbon-guru-tile');

        if (btn) {
            btn.classList.remove('cg-minimal');
        }
        if (header) {
            header.classList.remove('cg-header-minimal');
        }
        if (tile) {
            tile.classList.remove('cg-tile-minimal');
        }

        // Reset the timer
        this.startMinimalStateTimer();
    }

    isInMinimalState() {
        const btn = document.getElementById('carbon-analyze-btn');
        return btn && btn.classList.contains('cg-minimal');
    }

    restoreTileState() {
        // Restore tile state from extension-private storage (persists across page refreshes)
        browser.storage.local.get('carbonguru_tile_expanded').then(result => {
            if (result.carbonguru_tile_expanded === true) {
                void 0;
                // Use setTimeout to ensure DOM is fully ready
                setTimeout(() => {
                    this.expandTile();
                }, 200);
            } else {
                void 0;
            }
        });
    }

    showCurrentPage() {
        const currentPageDiv = document.getElementById('carbon-current-page');
        const statusSection = document.getElementById('carbon-status-section');
        
        if (currentPageDiv && statusSection) {
            const currentUrl = window.location.href;
            const pageUrlDiv = document.getElementById('carbon-page-url');
            
            if (pageUrlDiv) {
                pageUrlDiv.textContent = currentUrl;
                currentPageDiv.style.display = 'block';
                statusSection.style.display = 'block';
            }
        }
    }
    

    

    
    showProgress() {
        const source = this.getSourcePrefix();
        const itemType = this.getItemType();
        this.showStatus(`Processing ${source} ${itemType} items...`, {
            type: 'progress',
            showProgress: true
        });
        this.animateProgressBar();
        this.isAnalyzing = true;
        this.clearMinimalStateTimer();
    }

    hideProgress() {
        this.hideStatus();
        this.isAnalyzing = false;
        this.startMinimalStateTimer();
    }

    // Stage 1: Initial cart extraction - show items with prices (no footprints, no buttons)
    getCurrencySymbol(currencyCode) {
        // Map currency codes to symbols - complete list from exchange_rates database table (40 currencies)
        const currencyMap = {
            // Americas
            'USD': '$', 'CAD': 'C$', 'MXN': 'MX$', 'BRL': 'R$',
            'ARS': 'AR$', 'CLP': 'CLP$', 'COP': 'COL$', 'PEN': 'S/',
            // Europe
            'EUR': '€', 'GBP': '£', 'CHF': 'CHF', 'SEK': 'kr', 'NOK': 'kr',
            'DKK': 'kr', 'PLN': 'zł', 'CZK': 'Kč', 'HUF': 'Ft', 'RUB': '₽',
            'TRY': '₺', 'ILS': '₪', 'BGN': 'лв', 'HRK': 'kn', 'RON': 'lei',
            // Asia Pacific
            'JPY': '¥', 'CNY': '¥', 'INR': '₹', 'AUD': 'A$', 'NZD': 'NZ$',
            'SGD': 'S$', 'HKD': 'HK$', 'KRW': '₩', 'TWD': 'NT$',
            'THB': '฿', 'MYR': 'RM', 'PHP': '₱', 'IDR': 'Rp', 'VND': '₫',
            'PKR': '₨',
            // Middle East
            'AED': 'AED', 'SAR': 'SAR', 'QAR': 'QAR', 'KWD': 'KWD',
            'BHD': 'BHD', 'OMR': 'OMR', 'EGP': 'E£',
            // Africa
            'ZAR': 'R', 'NGN': '₦', 'KES': 'KSh', 'GHS': 'GH₵', 'MAD': 'MAD'
        };
        // Use currency code if no symbol mapping (better than defaulting to $)
        return currencyMap[currencyCode] || (currencyCode || '$');
    }

    escapeHtml(text) {
        // Pure-string HTML escaping (no DOM dependency, safe on custom DOMs like Gmail)
        if (typeof text !== 'string') return '';
        return text.replace(/[&<>"']/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[m]));
    }

    createPCFComparisonCard(matchData) {
        /**
         * Create a reusable PCF comparison card component.
         *
         * Args:
         *     matchData: PCF comparison match object from backend
         *
         * Returns:
         *     DOM element ready to insert into page
         */
        if (!matchData || !matchData.match) {
            return null;
        }

        const match = matchData.match;

        // Create container
        const card = document.createElement('div');
        card.className = 'carbon-pcf-comparison';

        // Build card HTML
        const descriptionHtml = match.description ? ` <span class="carbon-comparison-description">(${this.escapeHtml(match.description)})</span>` : '';
        card.innerHTML = `
            <div class="carbon-comparison-header">Similar footprint to:</div>
            <div class="carbon-comparison-content">
                <div class="carbon-comparison-product">
                    <div class="carbon-comparison-name">${this.escapeHtml(match.base_product_name)}${descriptionHtml}</div>
                    <div class="carbon-comparison-pcf">${match.base_pcf.toFixed(2)} kg CO₂e each</div>
                </div>
                <!-- DISABLED: Multiplier display disabled while using base products only -->
                <!-- <div class="carbon-comparison-multiplier">${match.multiplier}x</div> -->
            </div>
        `;

        return card;
    }

    showCartExtractionStatus(data) {
        void 0;
        void 0;
        void 0;
        void 0;
        void 0;

        // Mark cart processing as active - hide buttons for individual items
        this.cartProcessingActive = true;

        const statusSection = document.getElementById('carbon-status-section');
        const statusText = document.getElementById('carbon-status-text');
        const statusIcon = document.getElementById('carbon-status-icon');
        const resultsSection = document.getElementById('carbon-results-section');

        if (!statusText) return;

        // Clear any previous results section content to prevent stacking
        if (resultsSection) {
            resultsSection.style.display = 'none';
        }

        // Hide the status icon for card display
        if (statusIcon) {
            statusIcon.style.display = 'none';
        }

        // Use "Order" for order pages, "Shopping Cart" for shopping carts
        const cartOrOrder = data.analysis_type === 'order_page' ? 'Order' : 'Shopping Cart';
        const cartTitle = data.enhanced_display_title || data.enhanced_title || data.product_name || cartOrOrder;

        // Build items list with prices (no footprints yet - Stage 1)
        let itemsHtml = '';
        if (data.items && data.items.length > 0) {
            itemsHtml = '<div class="carbon-cart-items-list">';
            data.items.forEach((item, index) => {
                const itemName = item.product_name || item.name || `Item ${index + 1}`;
                const currencySymbol = this.getCurrencySymbol(item.currency || 'USD');

                // Handle free items with original price
                let itemPrice = '';
                if (item.price !== undefined && item.price !== null) {
                    const currentPrice = parseFloat(item.price);
                    const originalPrice = item.original_price ? parseFloat(item.original_price) : null;

                    if (currentPrice === 0 && originalPrice && originalPrice > 0) {
                        // Free item with original price - show both
                        itemPrice = `${currencySymbol}0.00 (${currencySymbol}${originalPrice.toFixed(2)} original)`;
                    } else {
                        // Regular priced item (including 0 without original price)
                        itemPrice = `${currencySymbol}${currentPrice.toFixed(2)}`;
                    }
                }

                itemsHtml += `
                    <div class="carbon-cart-item">
                        <span class="carbon-cart-item-name">${index + 1}. ${this.escapeHtml(itemName)}</span>
                        <span class="carbon-cart-item-details">${this.escapeHtml(itemPrice)}</span>
                    </div>
                `;
            });
            itemsHtml += '</div>';
        }

        statusText.innerHTML = `
            <div class="carbon-results-section">
                <div class="carbon-result-header">ℹ️ Analysis in progress</div>
                <div class="carbon-cart-title">${this.escapeHtml(cartTitle)}</div>
                ${itemsHtml}
            </div>
        `;

        if (statusSection) {
            statusSection.style.display = 'block';
        }
    }

    // Stage 2: Show items as they're processed with individual footprints
    showCartItemProgress(data) {
        void 0;

        // Clear status section to prevent stacking with previous messages
        const statusSection = document.getElementById('carbon-status-section');
        if (statusSection) {
            statusSection.style.display = 'none';
        }

        const currentItem = data.current_item;
        const itemsComplete = data.items_complete || 0;
        const totalItems = data.total_items || 0;

        if (currentItem) {
            // Show the individual item using styled card format (matching Analysis Complete)
            const itemName = currentItem.product_name || 'Product';
            const price = currentItem.price || 0;
            const footprint = currentItem.product_carbon_footprint || 0;
            const comparison = currentItem.pcf_comparison;
            const socialCost = currentItem.social_cost_usd;
            const currency = currentItem.currency || 'USD';
            const currencySymbol = this.getCurrencySymbol(currency);

            // Build classification info (NAICS or Open CEDA) - EXACT match to Analysis Complete format
            let classificationInfo = '';
            let methodInfo = '';

            if (currentItem.sector_code && currentItem.sector_name) {
                // Open CEDA classification
                classificationInfo = `Sector: ${currentItem.sector_code} - ${currentItem.sector_name}`;
                const countryCode = currentItem.country_code || 'Global';
                methodInfo = `Method: Open CEDA (${countryCode})`;
            } else if (currentItem.naics_code && currentItem.naics_title) {
                // NAICS classification
                classificationInfo = `NAICS: ${currentItem.naics_code} - ${currentItem.naics_title}`;
                methodInfo = `Method: US EIO`;
            } else if (currentItem.naics_code) {
                // NAICS code only
                classificationInfo = `NAICS: ${currentItem.naics_code}`;
                methodInfo = `Method: US EIO`;
            } else {
                // Fallback
                classificationInfo = 'Classification: N/A';
                methodInfo = `Method: ${currentItem.lca_type || 'EIO'}`;
            }

            // Build PCF comparison text if available
            let comparisonHtml = '';
            if (comparison && comparison.match) {
                const match = comparison.match;
                // DISABLED: Multiplier display disabled while using base products only
                // const multiplier = match.multiplier || 1;
                // const refProduct = match.base_product_name || 'reference product';
                // comparisonHtml = `<div class="carbon-pcf-comparison">≈ ${multiplier}x ${refProduct}</div>`;
            }

            // Display using showResults() to show green expandable tile (same as single product)
            // This will populate the tile with NAICS/Sector classification and method
            this.showResults({
                product_name: `✓ Item ${itemsComplete}/${totalItems}: ${itemName}`,
                product_carbon_footprint: footprint,
                price: price,
                currency: currency,
                social_cost_usd: socialCost,
                sector_code: currentItem.sector_code,
                sector_name: currentItem.sector_name,
                naics_code: currentItem.naics_code,
                naics_title: currentItem.naics_title,
                country_code: currentItem.country_code,
                pcf_comparison: comparison,
                lca_type: currentItem.lca_type || 'eio',
                // Don't show action buttons for individual cart items (only for final cart)
                details_url: null,
                // Mark as cart item to prevent button display
                analysis_type: 'cart_item'
            });
        } else {
            // Fallback: Show count-based progress
            const source = this.getSourcePrefix();
            const progressMessage = `<b>Processing ${source} cart items...</b>\n${itemsComplete}/${totalItems} items analyzed`;

            this.showStatus(progressMessage, {
                type: 'info',
                duration: 0,
                showCurrentPage: true
            });
        }
    }

    // Helper function to get currency symbol - complete list from exchange_rates database table (40 currencies)
    getCurrencySymbol(currency) {
        const symbols = {
            // Americas
            'USD': '$', 'CAD': 'C$', 'MXN': 'MX$', 'BRL': 'R$',
            'ARS': 'AR$', 'CLP': 'CLP$', 'COP': 'COL$', 'PEN': 'S/',
            // Europe
            'EUR': '€', 'GBP': '£', 'CHF': 'CHF', 'SEK': 'kr', 'NOK': 'kr',
            'DKK': 'kr', 'PLN': 'zł', 'CZK': 'Kč', 'HUF': 'Ft', 'RUB': '₽',
            'TRY': '₺', 'ILS': '₪', 'BGN': 'лв', 'HRK': 'kn', 'RON': 'lei',
            // Asia Pacific
            'JPY': '¥', 'CNY': '¥', 'INR': '₹', 'AUD': 'A$', 'NZD': 'NZ$',
            'SGD': 'S$', 'HKD': 'HK$', 'KRW': '₩', 'TWD': 'NT$',
            'THB': '฿', 'MYR': 'RM', 'PHP': '₱', 'IDR': 'Rp', 'VND': '₫',
            'PKR': '₨',
            // Middle East
            'AED': 'AED', 'SAR': 'SAR', 'QAR': 'QAR', 'KWD': 'KWD',
            'BHD': 'BHD', 'OMR': 'OMR', 'EGP': 'E£',
            // Africa
            'ZAR': 'R', 'NGN': '₦', 'KES': 'KSh', 'GHS': 'GH₵', 'MAD': 'MAD'
        };
        // Use currency code if no symbol mapping (better than defaulting to $)
        return symbols[currency.toUpperCase()] || (currency || '$');
    }

    // Stage 3: Analysis complete - show final footprint with item details and action buttons
    showCartCompletionStatus(data) {
        void 0;

        // End cart processing session - allow buttons for final cart summary
        this.cartProcessingActive = false;

        // Hide progress bar completely (it's a sibling element, not inside statusText)
        const progressBar = document.getElementById('carbon-progress-bar');
        if (progressBar) {
            progressBar.style.display = 'none';
        }
        if (this.progressInterval) {
            clearInterval(this.progressInterval);
        }

        // Immediately hide Stage 2 results - Stage 3 summary replaces everything
        const resultsSection = document.getElementById('carbon-results-section');
        if (resultsSection) {
            resultsSection.style.display = 'none';
        }

        const statusSection = document.getElementById('carbon-status-section');
        const statusText = document.getElementById('carbon-status-text');
        const statusIcon = document.getElementById('carbon-status-icon');

        if (!statusText) return;

        // Hide the status icon (no spinning icon for completion)
        if (statusIcon) {
            statusIcon.style.display = 'none';
        }

        // Store current analysis result for basket functionality
        this.currentAnalysisResult = data;

        // Use "Order" for order pages, "Shopping Cart" for shopping carts
        const cartOrOrder = data.analysis_type === 'order_page' ? 'Order' : 'Shopping Cart';
        const cartTitle = data.enhanced_display_title || data.enhanced_title || data.product_name || cartOrOrder;
        const totalFootprint = data.product_carbon_footprint ? `${data.product_carbon_footprint.toFixed(2)} kg CO₂e` : 'N/A';

        // Build items list with footprints and info icons
        let itemsHtml = '';
        const itemsWithComparisons = [];  // Track items for later comparison insertion
        if (data.items && data.items.length > 0) {
            itemsHtml = '<div class="carbon-cart-items-list">';
            data.items.forEach((item, index) => {
                const itemName = item.product_name || item.name || `Item ${index + 1}`;
                const currencySymbol = this.getCurrencySymbol(item.currency || 'USD');
                const itemPrice = item.price ? `${currencySymbol}${parseFloat(item.price).toFixed(2)}` : '';
                const itemFootprint = item.product_carbon_footprint ? `${item.product_carbon_footprint.toFixed(2)} kg CO₂e` : 'N/A';

                // Add info icon if item has PCF comparison
                const infoIcon = item.pcf_comparison ? '<span class="carbon-info-icon" data-item-index="' + index + '" title="View comparable product">ⓘ</span>' : '';

                itemsHtml += `
                    <div class="carbon-cart-item" data-item-index="${index}">
                        <span class="carbon-cart-item-name">${index + 1}. ${itemName}</span>
                        <span class="carbon-cart-item-details">${itemPrice} • ${itemFootprint} ${infoIcon}</span>
                    </div>
                `;
                // Track items with PCF comparisons for overlay
                if (item.pcf_comparison) {
                    itemsWithComparisons.push({ index, item });
                }
            });
            itemsHtml += '</div>';
        }

        // Build PCF comparison card HTML for cart total (use cart_pcf_comparison)
        let cartComparisonHtml = '';
        if (data.cart_pcf_comparison && data.cart_pcf_comparison.match) {
            const match = data.cart_pcf_comparison.match;
            const descriptionHtml = match.description ? ` <span class="carbon-comparison-description">(${this.escapeHtml(match.description)})</span>` : '';
            cartComparisonHtml = `
                <div class="carbon-pcf-comparison">
                    <div class="carbon-comparison-header">Cart footprint similar to:</div>
                    <div class="carbon-comparison-content">
                        <div class="carbon-comparison-product">
                            <div class="carbon-comparison-name">${this.escapeHtml(match.base_product_name)}${descriptionHtml}</div>
                            <div class="carbon-comparison-pcf">${match.base_pcf.toFixed(2)} kg CO₂e each</div>
                        </div>
                        <!-- DISABLED: Multiplier display disabled while using base products only -->
                        <!-- ${match.multiplier > 1 ? `<div class="carbon-comparison-multiplier">${match.multiplier}x</div>` : ''} -->
                    </div>
                </div>
            `;
        }

        // DISABLED: Offset cost badge - buttons show dynamic pricing instead
        const offsetCostHtml = '';
        const carbonCostHtml = '';

        // Build cart title - will be made into a link after DOM update
        const cartTitleHtml = `<div class="carbon-cart-title" id="carbon-cart-title">${this.escapeHtml(cartTitle)}</div>`;

        statusText.innerHTML = `
            <div class="carbon-results-section">
                <div class="carbon-result-header">✅ Analysis Complete</div>
                ${cartTitleHtml}
                <div class="carbon-footprint-display">
                    <div class="carbon-footprint-value">${totalFootprint}</div>
                    ${offsetCostHtml}
                    ${carbonCostHtml}
                </div>
                ${cartComparisonHtml}
                ${itemsHtml}
                <div class="carbon-action-buttons">
                    <div class="carbon-action-buttons-row">
                        <button id="carbon-cart-offset-btn" class="carbon-offset-btn" style="display: none;">
                            <span class="btn-label">Offset:</span>
                            <span class="btn-cost">...</span>
                        </button>
                        <button id="carbon-cart-removal-btn" class="carbon-removal-btn" style="display: none;">
                            <span class="btn-label">Remove:</span>
                            <span class="btn-cost">...</span>
                        </button>
                    </div>
                    <!-- DISABLED: View Details button - cart title is now linked instead -->
                </div>
            </div>
        `;

        if (statusSection) {
            statusSection.style.display = 'block';
            statusSection.className = 'carbon-status-section';
        }

        // Setup info icon overlay handlers for individual items (after DOM is updated)
        if (itemsWithComparisons.length > 0) {
            setTimeout(() => this.setupPcfOverlay(itemsWithComparisons), 100);
        }

        // Setup button listeners for cart actions
        this.setupCartCompletionButtons(data);
    }

    // showOrderCompletionStatus() DELETED - Orders now use showCartCompletionStatus() (same display as carts)

    async setupCartCompletionButtons(data) {
        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
        const offsetBtn = document.getElementById('carbon-cart-offset-btn');
        const removalBtn = document.getElementById('carbon-cart-removal-btn');
        const cartTitleEl = document.getElementById('carbon-cart-title');

        // Link cart title to WooCommerce cart summary URL
        // PRIORITY: Use backend-provided URL first (handles WordPress post existence check)
        // Only fall back to constructing URL if backend didn't provide one
        if (cartTitleEl && data.session_id) {
            try {
                // Check for backend-provided URL (preferred - includes WordPress post existence check)
                let cartUrl = data.cart_summary_url || data.details_url || data.wc_product_url;

                if (!cartUrl) {
                    // Fallback: construct URL (legacy behavior)
                    const detailsUrl = await runtime.sendMessage({
                        action: 'getCartSummaryUrl',
                        sessionId: data.session_id,
                        analysisType: data.analysis_type
                    });
                    cartUrl = detailsUrl?.url;
                }

                if (cartUrl) {
                    // Ensure URL is absolute (prepend backendUrl if relative)
                    if (cartUrl.startsWith('/')) {
                        const backendUrl = await runtime.sendMessage({ action: 'getBackendUrl' });
                        cartUrl = (backendUrl?.url || 'https://carbonguru.io') + cartUrl;
                    }
                    const cartTitle = data.enhanced_display_title || data.product_name || 'Order';
                    cartTitleEl.innerHTML = `<a href="${encodeURI(cartUrl)}" target="_blank" class="carbon-product-name-link">${this.escapeHtml(cartTitle)}</a>`;
                }
            } catch (error) {
                void 0;
            }
        }

        // Fetch costs and setup offset/removal buttons (same pattern as individual products)
        if ((offsetBtn || removalBtn) && data.product_carbon_footprint) {
            const carbon_kg = data.product_carbon_footprint;

            // Show buttons with loading state
            if (offsetBtn) {
                offsetBtn.querySelector('.btn-cost').textContent = '...';
                offsetBtn.style.display = 'inline-flex';
            }
            if (removalBtn) {
                removalBtn.querySelector('.btn-cost').textContent = '...';
                removalBtn.style.display = 'inline-flex';
            }

            // Fetch costs from background script with retry (service worker may not be ready immediately)
            const maxRetries = 3;
            for (let attempt = 1; attempt <= maxRetries; attempt++) {
                try {
                    const costs = await runtime.sendMessage({
                        action: 'getOffsetCosts',
                        carbon_kg: carbon_kg
                    });

                    if (costs && costs.success) {
                        if (offsetBtn && costs.offset_cost_usd > 0) {
                            offsetBtn.querySelector('.btn-cost').textContent = formatCost(costs.offset_cost_usd);
                        } else if (offsetBtn) {
                            offsetBtn.querySelector('.btn-cost').textContent = 'N/A';
                        }

                        if (removalBtn && costs.removal_cost_usd > 0) {
                            removalBtn.querySelector('.btn-cost').textContent = formatCost(costs.removal_cost_usd);
                        } else if (removalBtn) {
                            removalBtn.querySelector('.btn-cost').textContent = 'N/A';
                        }
                        break; // Success - exit retry loop
                    } else if (attempt < maxRetries) {
                        await new Promise(r => setTimeout(r, 500)); // Wait 500ms before retry
                    } else {
                        if (offsetBtn) offsetBtn.querySelector('.btn-cost').textContent = 'now';
                        if (removalBtn) removalBtn.querySelector('.btn-cost').textContent = 'now';
                    }
                } catch (error) {
                    if (attempt < maxRetries) {
                        void 0;
                        await new Promise(r => setTimeout(r, 500)); // Wait 500ms before retry
                    } else {
                        console.error('CarbonGuru: Error fetching cart costs after retries:', error);
                        if (offsetBtn) offsetBtn.querySelector('.btn-cost').textContent = 'now';
                        if (removalBtn) removalBtn.querySelector('.btn-cost').textContent = 'now';
                    }
                }
            }

            // Setup click handlers for offset button
            if (offsetBtn) {
                offsetBtn.onclick = async () => {
                    const originalCost = offsetBtn.querySelector('.btn-cost').textContent;
                    try {
                        offsetBtn.disabled = true;
                        offsetBtn.querySelector('.btn-cost').textContent = '...';

                        // Add all cart items with offset preference
                        const response = await this.addCartItemsWithType(data, 'offset');

                        if (response && response.success) {
                            // Show count of items added (matches recent items behavior)
                            const countText = response.added > 1 ? `✓ ${response.added} items` : '✓ Added!';
                            offsetBtn.querySelector('.btn-cost').textContent = countText;
                            offsetBtn.classList.add('carbon-basket-added');
                        } else {
                            offsetBtn.querySelector('.btn-cost').textContent = 'Error';
                            setTimeout(() => {
                                if (offsetBtn.querySelector('.btn-cost').textContent === 'Error') {
                                    offsetBtn.querySelector('.btn-cost').textContent = originalCost;
                                }
                            }, 3000);
                        }
                        offsetBtn.disabled = false;
                    } catch (error) {
                        console.error('CarbonGuru: Error adding cart to basket:', error);
                        offsetBtn.disabled = false;
                        offsetBtn.querySelector('.btn-cost').textContent = 'Error';
                        setTimeout(() => {
                            if (offsetBtn.querySelector('.btn-cost').textContent === 'Error') {
                                offsetBtn.querySelector('.btn-cost').textContent = originalCost;
                            }
                        }, 3000);
                    }
                };
            }

            // Setup click handlers for removal button
            if (removalBtn) {
                removalBtn.onclick = async () => {
                    const originalCost = removalBtn.querySelector('.btn-cost').textContent;
                    try {
                        removalBtn.disabled = true;
                        removalBtn.querySelector('.btn-cost').textContent = '...';

                        // Add all cart items with removal preference
                        const response = await this.addCartItemsWithType(data, 'removal');

                        if (response && response.success) {
                            // Show count of items added (matches recent items behavior)
                            const countText = response.added > 1 ? `✓ ${response.added} items` : '✓ Added!';
                            removalBtn.querySelector('.btn-cost').textContent = countText;
                            removalBtn.classList.add('carbon-basket-added');
                        } else {
                            removalBtn.querySelector('.btn-cost').textContent = 'Error';
                            setTimeout(() => {
                                if (removalBtn.querySelector('.btn-cost').textContent === 'Error') {
                                    removalBtn.querySelector('.btn-cost').textContent = originalCost;
                                }
                            }, 3000);
                        }
                        removalBtn.disabled = false;
                    } catch (error) {
                        console.error('CarbonGuru: Error adding cart to basket:', error);
                        removalBtn.disabled = false;
                        removalBtn.querySelector('.btn-cost').textContent = 'Error';
                        setTimeout(() => {
                            if (removalBtn.querySelector('.btn-cost').textContent === 'Error') {
                                removalBtn.querySelector('.btn-cost').textContent = originalCost;
                            }
                        }, 3000);
                    }
                };
            }
        }
    }

    // Helper to add all cart items with a specific preference type
    async addCartItemsWithType(cartData, preferenceType) {
        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;

        if (!cartData.items || cartData.items.length === 0) {
            return { success: false, error: 'No items in cart' };
        }

        let successCount = 0;
        for (const item of cartData.items) {
            try {
                const response = await runtime.sendMessage({
                    action: 'addToBasketWithType',
                    item: {
                        carbonguru_id: item.carbonguru_id || item.id,
                        product_name: item.product_name || item.name,
                        product_carbon_footprint: item.product_carbon_footprint,
                        details_url: item.details_url
                    },
                    preferenceType: preferenceType
                });
                if (response && response.success) {
                    successCount++;
                }
            } catch (error) {
                console.error('CarbonGuru: Error adding cart item:', error);
            }
        }

        return { success: successCount > 0, added: successCount, total: cartData.items.length };
    }

    async showResults(data) {
        void 0;

        this.hideProgress();

        // Hide status section to prevent message stacking with results
        const statusSection = document.getElementById('carbon-status-section');
        if (statusSection) {
            statusSection.style.display = 'none';
        }

        // Check if this is a cart result (including order pages)
        const isCartResult = data.analysis_type === 'shopping_cart' || data.analysis_type === 'order_page' || data.type === 'cart';

        // Cart/order SUMMARIES (Stage 3: analysis_complete) should use showCartCompletionStatus() ONLY
        // But Stage 2 individual cart items (content_type: individual_item_complete) legitimately use showResults
        // This matches the pattern used for email orders - summaries skip showResults entirely
        if (isCartResult && data.content_type === 'analysis_complete') {
            void 0;
            return;
        }

        // Check if this is an individual cart item during processing (Stage 2)
        // Don't show action buttons for cart items - only for final cart summary (Stage 3)
        const isCartItem = data.analysis_type === 'cart_item';

        // SIMPLE FIX: If we're processing a cart (not final summary), hide all buttons
        // This catches ALL individual item results during cart processing
        const hideButtons = (isCartResult === false && this.cartProcessingActive === true) || isCartItem;
        void 0;

        const resultsSection = document.getElementById('carbon-results-section');
        const footprintDisplay = document.getElementById('carbon-footprint-display');
        const productDetails = document.getElementById('carbon-product-details');
        // DISABLED: View Details button - product name is now linked instead
        // const viewDetailsBtn = document.getElementById('carbon-view-details-btn');
        const offsetBtn = document.getElementById('carbon-offset-btn');
        const removalBtn = document.getElementById('carbon-removal-btn');

        // For carts, skip the expandable tile and only show status message (at end of this function)
        // For individual products and cart items, show the expandable results tile
        // Cart items (Stage 2) WILL show the green tile with footprint data, just without action buttons
        if (resultsSection && !isCartResult) resultsSection.style.display = 'block';
        
        if (footprintDisplay && data.product_carbon_footprint) {
            // Determine if this is a cart or individual product for appropriate labeling
            // isCartResult already declared at line 2034
            const footprintLabel = isCartResult ? 'Total Cart Footprint' : 'Carbon Footprint';
            const itemInfo = isCartResult && data.total_items ? ` (${data.total_items} items)` : '';

            // Add Carbon Cost display if available (EPA 2023 Social Cost of Carbon)
            const offsetCostValue = data.offset_cost_usd ?? data.display_cost_usd ?? data.orderable_cost_usd;
            const parsedOffsetCost = offsetCostValue !== undefined && offsetCostValue !== null ? parseFloat(offsetCostValue) : null;
            const carbonCostValue = data.social_cost_usd !== undefined && data.social_cost_usd !== null ? parseFloat(data.social_cost_usd) : null;

            // DISABLED: Offset cost badge - buttons show dynamic pricing instead
            // const offsetCostHtml = parsedOffsetCost && parsedOffsetCost > 0
            //     ? `<div class="carbon-cost-label">Offset Cost: $${parsedOffsetCost.toFixed(2)} USD</div>`
            //     : '';
            const offsetCostHtml = ''; // Disabled - offset/removal buttons show pricing

            // DISABLED: Social cost of carbon display - uncomment to re-enable
            // const carbonCostHtml = carbonCostValue && carbonCostValue > 0
            //     ? `<div class="carbon-cost-label">Carbon Cost: $${carbonCostValue.toFixed(2)} USD</div>`
            //     : '';
            const carbonCostHtml = ''; // Disabled - social cost not shown

            footprintDisplay.innerHTML = `
                <div class="carbon-footprint-value">${data.product_carbon_footprint.toFixed(2)} kg CO₂e</div>
                <div class="carbon-footprint-label">${footprintLabel}${itemInfo}</div>
                ${offsetCostHtml}
                ${carbonCostHtml}
            `;
        }
        
        if (productDetails && (data.product_name || data.enhanced_display_title)) {
            // Determine if this is a cart or individual product
            // isCartResult already declared at line 2034
            const displayTitle = data.enhanced_display_title || data.product_name;
            
            // Determine classification type and method based on available data
            let classificationInfo = '';
            let methodInfo = '';
            
            if (isCartResult) {
                // Cart/Order-specific display - use total_items from API response
                const itemCount = data.total_items || (data.items ? data.items.length : 0);
                const cartContext = data.cart_optimization_context || {};
                const cartOrOrder = data.analysis_type === 'order_page' ? 'Order' : 'Shopping Cart';
                classificationInfo = `${data.analysis_type === 'order_page' ? '📦' : '🛒'} ${cartOrOrder} Analysis • ${itemCount} item${itemCount !== 1 ? 's' : ''}`;
                methodInfo = `${cartOrOrder} Optimization: ${cartContext.seller_name || 'Enhanced'} Processing`;
            } else if (data.sector_code && data.sector_name) {
                // Open CEDA / Global processing
                classificationInfo = `Sector: ${data.sector_code} - ${data.sector_name}`;
                methodInfo = `Method: Open CEDA (${data.country_code || 'Global'})`;
            } else if (data.naics_code && data.naics_title) {
                // US NAICS / EIO processing  
                classificationInfo = `NAICS: ${data.naics_code} - ${data.naics_title}`;
                methodInfo = `Method: US EIO`;
            } else if (data.naics_code) {
                // NAICS code only
                classificationInfo = `NAICS: ${data.naics_code}`;
                methodInfo = `Method: US EIO`;
            } else {
                // Fallback
                classificationInfo = 'Classification: N/A';
                methodInfo = `Method: ${data.lca_type || 'EIO'}`;
            }
            
            const titleClass = isCartResult ? 'carbon-cart-name' : 'carbon-product-name';

            // Build PCF comparison display if available (same format as Recent Items)
            let pcfComparisonHtml = '';
            if (data.pcf_comparison && data.pcf_comparison.match) {
                const match = data.pcf_comparison.match;
                const descriptionHtml = match.description ? ` <span class="carbon-comparison-description">(${this.escapeHtml(match.description)})</span>` : '';
                pcfComparisonHtml = `
                    <div class="carbon-pcf-comparison">
                        <div class="carbon-comparison-header">Similar footprint to:</div>
                        <div class="carbon-comparison-content">
                            <div class="carbon-comparison-product">
                                <div class="carbon-comparison-name">${this.escapeHtml(match.base_product_name)}${descriptionHtml}</div>
                                <div class="carbon-comparison-pcf">${match.base_pcf.toFixed(2)} kg CO₂e each</div>
                            </div>
                            <!-- DISABLED: Multiplier display disabled while using base products only -->
                            <!-- ${match.multiplier > 1 ? `<div class="carbon-comparison-multiplier">${match.multiplier}x</div>` : ''} -->
                        </div>
                    </div>
                `;
            }

            // Render product name initially (may be updated to link when URL is ready)
            productDetails.innerHTML = `
                <div class="${titleClass}" id="carbon-product-title">${displayTitle}</div>
                <div class="carbon-product-info">
                    <span class="carbon-classification">${classificationInfo}</span>
                    <span class="carbon-method">${methodInfo}</span>
                </div>
                ${pcfComparisonHtml}
            `;
        }

        // DISABLED: View Details button - product name is now linked instead
        // if (viewDetailsBtn) {
        //     viewDetailsBtn.style.display = 'none';
        // }

        // Link product name to details URL (replaces View Details button)
        if (data.details_url && !hideButtons) {
            const productTitleEl = document.getElementById('carbon-product-title');
            let finalUrl;

            // Helper to update product name as clickable link
            const makeProductNameLink = (url) => {
                if (productTitleEl && url) {
                    const displayTitle = data.enhanced_display_title || data.product_name;
                    productTitleEl.innerHTML = `<a href="${encodeURI(url)}" target="_blank" class="carbon-product-name-link">${this.escapeHtml(displayTitle)}</a>`;
                }
            };

            // Check for WooCommerce URL first (highest priority)
            if (data.wc_product_url) {
                finalUrl = data.wc_product_url;
                void 0;
                makeProductNameLink(finalUrl);
            }
            // Check if this is a cart session (details_url starts with /sessions/)
            else if (data.details_url.startsWith('/sessions/')) {
                // For cart sessions, link to shopping cart summary
                const sessionId = data.details_url.replace('/sessions/', '');
                finalUrl = await this.getCartSummaryUrl(sessionId);
                void 0;
                makeProductNameLink(finalUrl);
            }
            // Internal product URLs - WooCommerce sync not complete yet, poll for URL
            else if (data.details_url.startsWith('/product/') && !data.details_url.startsWith('https://')) {
                void 0;

                // Immediately link to the internal product URL so the title is always clickable
                const internalUrl = await this.getProductUrl(data.details_url);
                makeProductNameLink(internalUrl);

                // Poll for WooCommerce URL every 500ms for up to 30 seconds to upgrade the link
                let pollCount = 0;
                const maxPolls = 60; // 60 polls × 500ms = 30 seconds
                const pollInterval = setInterval(async () => {
                    pollCount++;

                    if (pollCount > maxPolls) {
                        void 0;
                        clearInterval(pollInterval);
                        return;
                    }

                    try {
                        // Use background script to get recent items (same method as Recent Items list)
                        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
                        const response = await runtime.sendMessage({
                            action: 'getRecentItems',
                            limit: 10
                        });

                        if (!response || !response.success || !response.items) {
                            return; // Keep polling
                        }

                        // Find this item in the response
                        const thisItem = response.items.find(i => i.carbonguru_id === data.carbonguru_id);

                        if (thisItem?.wc_product_url) {
                            void 0;

                            // Update product name to be a link
                            makeProductNameLink(thisItem.wc_product_url);

                            // Refresh Recent Items list to show WooCommerce URLs
                            void 0;
                            this.loadRecentItems();

                            clearInterval(pollInterval);
                        }
                    } catch (error) {
                        void 0;
                    }
                }, 500); // Poll every 500ms
            } else {
                // For individual products, use the standard product details URL
                finalUrl = await this.getProductUrl(data.details_url);
                void 0;
                makeProductNameLink(finalUrl);
            }
        }
        // Fallback: details_url missing but we can construct a link from session/carbonguru IDs or source URL
        else if (!hideButtons) {
            const productTitleEl = document.getElementById('carbon-product-title');
            if (productTitleEl) {
                const displayTitle = data.enhanced_display_title || data.product_name;
                let fallbackUrl = null;

                // Try to construct internal product URL from available IDs
                if (data.carbonguru_id && (data.session_id || data.id)) {
                    const sessionId = data.session_id || data.id;
                    fallbackUrl = await this.getProductUrl(`/product/${sessionId}/${data.carbonguru_id}`);
                }
                // Use source page URL as last resort
                else if (data.url || data.source_url) {
                    fallbackUrl = data.url || data.source_url;
                }

                if (fallbackUrl && displayTitle) {
                    productTitleEl.innerHTML = `<a href="${encodeURI(fallbackUrl)}" target="_blank" class="carbon-product-name-link">${this.escapeHtml(displayTitle)}</a>`;
                    void 0;
                }
            }
        }

        // Offset and Removal buttons - fetch costs and add to basket
        if ((offsetBtn || removalBtn) && !hideButtons) {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;

            // Fetch costs for both buttons
            if (data.product_carbon_footprint) {
                const carbon_kg = data.product_carbon_footprint;

                // Deduplicate concurrent cost fetches (e.g. individual_item_complete + analysisComplete)
                // Use a generation counter to detect when a newer showResults call supersedes this one
                this._costFetchGeneration = (this._costFetchGeneration || 0) + 1;
                const thisGeneration = this._costFetchGeneration;

                // Show buttons with loading state
                if (offsetBtn) {
                    offsetBtn.querySelector('.btn-cost').textContent = '...';
                    offsetBtn.style.display = 'inline-block';
                }
                if (removalBtn) {
                    removalBtn.querySelector('.btn-cost').textContent = '...';
                    removalBtn.style.display = 'inline-block';
                }

                // Fetch costs from background script with retry (service worker may not be ready immediately)
                const maxRetries = 3;
                for (let attempt = 1; attempt <= maxRetries; attempt++) {
                    // Abort if a newer showResults call has started
                    if (this._costFetchGeneration !== thisGeneration) {
                        void 0;
                        break;
                    }

                    try {
                        const costs = await runtime.sendMessage({
                            action: 'getOffsetCosts',
                            carbon_kg: carbon_kg
                        });

                        // Abort if superseded while waiting for response
                        if (this._costFetchGeneration !== thisGeneration) {
                            void 0;
                            break;
                        }

                        void 0;

                        if (costs && costs.success) {
                            // Update offset button with cost
                            if (offsetBtn && costs.offset_cost_usd > 0) {
                                offsetBtn.querySelector('.btn-cost').textContent = formatCost(costs.offset_cost_usd);
                            } else if (offsetBtn) {
                                offsetBtn.querySelector('.btn-cost').textContent = 'N/A';
                            }

                            // Update removal button with cost
                            if (removalBtn && costs.removal_cost_usd > 0) {
                                removalBtn.querySelector('.btn-cost').textContent = formatCost(costs.removal_cost_usd);
                            } else if (removalBtn) {
                                removalBtn.querySelector('.btn-cost').textContent = 'N/A';
                            }
                            break; // Success - exit retry loop
                        } else if (attempt < maxRetries) {
                            // Longer delay on first retry to give service worker time to wake up
                            const delay = attempt === 1 ? 1000 : 500;
                            await new Promise(r => setTimeout(r, delay));
                        } else {
                            // Fallback to generic labels on final failure
                            if (offsetBtn) offsetBtn.querySelector('.btn-cost').textContent = 'now';
                            if (removalBtn) removalBtn.querySelector('.btn-cost').textContent = 'now';
                            void 0;
                        }
                    } catch (error) {
                        if (attempt < maxRetries) {
                            void 0;
                            // Longer delay on first retry to give service worker time to wake up
                            const delay = attempt === 1 ? 1000 : 500;
                            await new Promise(r => setTimeout(r, delay));
                        } else {
                            console.error('CarbonGuru: Error fetching costs after retries:', error);
                            if (offsetBtn) offsetBtn.querySelector('.btn-cost').textContent = 'now';
                            if (removalBtn) removalBtn.querySelector('.btn-cost').textContent = 'now';
                        }
                    }
                }
            }

            // Setup click handlers to add to basket
            if (offsetBtn) {
                offsetBtn.onclick = async () => {
                    // Store original cost BEFORE try block so it's accessible in catch
                    const originalCost = offsetBtn.querySelector('.btn-cost').textContent;

                    try {
                        const item = this.currentAnalysisResult;
                        if (!item) {
                            console.error('CarbonGuru: No analysis result to add to basket');
                            return;
                        }

                        offsetBtn.disabled = true;
                        offsetBtn.querySelector('.btn-cost').textContent = '...';

                        const response = await runtime.sendMessage({
                            action: 'addToBasketWithType',
                            item: {
                                carbonguru_id: item.carbonguru_id,
                                product_name: item.product_name,
                                product_carbon_footprint: item.product_carbon_footprint,
                                details_url: item.details_url
                            },
                            preferenceType: 'offset'
                        });

                        if (response && response.success) {
                            // Show "Added!" for single items (matches recent items behavior)
                            offsetBtn.querySelector('.btn-cost').textContent = 'Added!';
                            offsetBtn.classList.add('carbon-basket-added');
                        } else {
                            offsetBtn.querySelector('.btn-cost').textContent = 'Error';
                            setTimeout(() => {
                                if (offsetBtn.querySelector('.btn-cost').textContent === 'Error') {
                                    offsetBtn.querySelector('.btn-cost').textContent = originalCost;
                                }
                            }, 3000);
                        }

                        offsetBtn.disabled = false;
                    } catch (error) {
                        console.error('CarbonGuru: Error adding to basket:', error);
                        offsetBtn.disabled = false;
                        offsetBtn.querySelector('.btn-cost').textContent = 'Error';
                        // Reset after error
                        setTimeout(() => {
                            if (offsetBtn.querySelector('.btn-cost').textContent === 'Error') {
                                offsetBtn.querySelector('.btn-cost').textContent = originalCost;
                            }
                        }, 3000);
                    }
                };
            }

            if (removalBtn) {
                removalBtn.onclick = async () => {
                    // Store original cost BEFORE try block so it's accessible in catch
                    const originalCost = removalBtn.querySelector('.btn-cost').textContent;

                    try {
                        const item = this.currentAnalysisResult;
                        if (!item) {
                            console.error('CarbonGuru: No analysis result to add to basket');
                            return;
                        }

                        removalBtn.disabled = true;
                        removalBtn.querySelector('.btn-cost').textContent = '...';

                        const response = await runtime.sendMessage({
                            action: 'addToBasketWithType',
                            item: {
                                carbonguru_id: item.carbonguru_id,
                                product_name: item.product_name,
                                product_carbon_footprint: item.product_carbon_footprint,
                                details_url: item.details_url
                            },
                            preferenceType: 'removal'
                        });

                        if (response && response.success) {
                            // Show "Added!" for single items (matches recent items behavior)
                            removalBtn.querySelector('.btn-cost').textContent = 'Added!';
                            removalBtn.classList.add('carbon-basket-added');
                        } else {
                            removalBtn.querySelector('.btn-cost').textContent = 'Error';
                            setTimeout(() => {
                                if (removalBtn.querySelector('.btn-cost').textContent === 'Error') {
                                    removalBtn.querySelector('.btn-cost').textContent = originalCost;
                                }
                            }, 3000);
                        }

                        removalBtn.disabled = false;
                    } catch (error) {
                        console.error('CarbonGuru: Error adding to basket:', error);
                        removalBtn.disabled = false;
                        removalBtn.querySelector('.btn-cost').textContent = 'Error';
                        // Reset after error
                        setTimeout(() => {
                            if (removalBtn.querySelector('.btn-cost').textContent === 'Error') {
                                removalBtn.querySelector('.btn-cost').textContent = originalCost;
                            }
                        }, 3000);
                    }
                };
            }
        }
        
        // Add to Basket button - hide during cart processing (Stage 2)
        const basketBtn = document.getElementById('carbon-add-to-basket-btn');
        if (basketBtn && !hideButtons) {
            basketBtn.style.display = 'inline-block';
        }
        
        // Store current analysis result for basket functionality
        this.currentAnalysisResult = data;

        // Show Analysis Complete status message for individual products only
        // NOTE: Cart/order summaries are now handled by showCartCompletionStatus() function
        // This showResults() function should ONLY be called for individual products

        if (!isCartResult) {
            // For individual products: populate PCF comparison in results card
            // The expandable results tile shows detailed information

            // Debug: Log what data we received
            void 0;
        }

        // Note: loadRecentItems() is called with delay in the completion handler
        // after showResults() completes (see analysisComplete message handler)
    }
    
    hideResults() {
        const resultsSection = document.getElementById('carbon-results-section');
        if (resultsSection) {
            resultsSection.style.display = 'none';
        }
    }

    setupPcfOverlay(itemsWithComparisons) {
        // Add event listeners to info icons for PCF comparison overlay
        const infoIcons = document.querySelectorAll('.carbon-info-icon');

        infoIcons.forEach(icon => {
            const itemIndex = parseInt(icon.getAttribute('data-item-index'));
            const itemData = itemsWithComparisons.find(i => i.index === itemIndex);

            if (itemData && itemData.item.pcf_comparison) {
                // Click to show overlay
                icon.addEventListener('click', (e) => {
                    e.stopPropagation();
                    this.showPcfOverlay(itemData.item, icon);
                });

                // Hover to show overlay
                icon.addEventListener('mouseenter', () => {
                    this.showPcfOverlay(itemData.item, icon);
                });
            }
        });
    }

    showPcfOverlay(item, triggeringIcon) {
        // Remove existing overlay
        const existingOverlay = document.getElementById('carbon-pcf-overlay');
        if (existingOverlay) existingOverlay.remove();

        const match = item.pcf_comparison.match;
        const descriptionHtml = match.description ? ` <span class="carbon-comparison-description">(${this.escapeHtml(match.description)})</span>` : '';
        const overlay = document.createElement('div');
        overlay.id = 'carbon-pcf-overlay';
        overlay.className = 'carbon-pcf-overlay-positioned';
        overlay.innerHTML = `
            <div class="carbon-pcf-overlay-content">
                <div class="carbon-comparison-header">Similar footprint to:</div>
                <div class="carbon-comparison-content">
                    <div class="carbon-comparison-product">
                        <div class="carbon-comparison-name">${this.escapeHtml(match.base_product_name)}${descriptionHtml}</div>
                        <div class="carbon-comparison-pcf">${match.base_pcf.toFixed(2)} kg CO₂e each</div>
                    </div>
                    <!-- DISABLED: Multiplier display disabled while using base products only -->
                    <!-- ${match.multiplier > 1 ? `<div class="carbon-comparison-multiplier">${match.multiplier}x</div>` : ''} -->
                </div>
                <div class="carbon-comparison-match">
                    <!-- DISABLED: Total footprint redundant when showing base products only -->
                    <!-- <span class="carbon-comparison-total">${match.total_pcf.toFixed(2)} kg CO₂e</span> -->
                    <span class="carbon-comparison-accuracy">${(match.match_score * 100).toFixed(1)}% match</span>
                </div>
            </div>
        `;

        document.body.appendChild(overlay);

        // Position overlay above and right-aligned to the icon
        if (triggeringIcon) {
            const iconRect = triggeringIcon.getBoundingClientRect();
            const overlayContent = overlay.querySelector('.carbon-pcf-overlay-content');

            // Position above the icon, right-aligned, vertically centered on icon
            const right = window.innerWidth - iconRect.right;
            const bottom = window.innerHeight - iconRect.top + 8; // 8px gap above icon

            overlay.style.position = 'fixed';
            overlay.style.right = `${right}px`;
            overlay.style.bottom = `${bottom}px`;
            overlay.style.background = 'transparent';
            overlay.style.pointerEvents = 'auto';

            // Allow content to be interactive
            overlayContent.style.pointerEvents = 'auto';
        }

        // Close on click outside or Escape key
        const closeOverlay = () => {
            overlay.remove();
        };

        overlay.addEventListener('click', (e) => {
            if (e.target === overlay) closeOverlay();
        });

        overlay.addEventListener('mouseleave', closeOverlay);

        document.addEventListener('keydown', function escHandler(e) {
            if (e.key === 'Escape') {
                closeOverlay();
                document.removeEventListener('keydown', escHandler);
            }
        });

        // Auto-close after 5 seconds
        setTimeout(closeOverlay, 5000);
    }

    setupRecentItemsViewAndOffsetButtons() {
        // No more View Details buttons - item names are now clickable links
        // Offset/Removal buttons are handled by setupBasketButtonListeners
    }

    /**
     * Fetch offset and removal costs for recent items and update button displays
     */
    async fetchRecentItemCosts() {
        void 0;

        if (!this.recentItemsData || this.recentItemsData.length === 0) {
            void 0;
            return;
        }

        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
        const recentItemsContainer = document.getElementById('carbon-recent-items');
        if (!recentItemsContainer) {
            void 0;
            return;
        }

        // Limit to first 10 items for performance
        const itemsToFetch = this.recentItemsData.slice(0, 10);
        void 0;

        for (let index = 0; index < itemsToFetch.length; index++) {
            const item = itemsToFetch[index];
            const carbon_kg = item.product_carbon_footprint;

            if (!carbon_kg || carbon_kg <= 0) {
                void 0;
                continue;
            }

            try {
                void 0;
                const costs = await runtime.sendMessage({
                    action: 'getOffsetCosts',
                    carbon_kg: carbon_kg
                });

                void 0;

                if (costs && costs.success) {
                    // Find buttons for this item by index
                    const offsetBtn = recentItemsContainer.querySelector(`.carbon-offset-btn[data-item-index="${index}"]`);
                    const removalBtn = recentItemsContainer.querySelector(`.carbon-removal-btn[data-item-index="${index}"]`);

                    void 0;

                    if (offsetBtn) {
                        const costSpan = offsetBtn.querySelector('.btn-cost');
                        if (costSpan && costs.offset_cost_usd > 0) {
                            costSpan.textContent = formatCost(costs.offset_cost_usd);
                            void 0;
                        } else if (costSpan) {
                            costSpan.textContent = 'N/A';
                        }
                    }

                    if (removalBtn) {
                        const costSpan = removalBtn.querySelector('.btn-cost');
                        if (costSpan && costs.removal_cost_usd > 0) {
                            costSpan.textContent = formatCost(costs.removal_cost_usd);
                            void 0;
                        } else if (costSpan) {
                            costSpan.textContent = 'N/A';
                        }
                    }
                } else {
                    void 0;
                }
            } catch (error) {
                void 0;
            }
        }
        void 0;
    }

    setupRecentItemsPcfOverlay() {
        // Add event listeners to info icons in Recent Items for PCF comparison overlay
        const infoIcons = document.querySelectorAll('.carbon-recent-info-icon');

        infoIcons.forEach(icon => {
            const itemId = icon.getAttribute('data-recent-item-id');
            const itemData = this.recentItemsWithComparisons?.find(i =>
                (i.carbonguru_id === itemId || i.id === itemId)
            );

            if (itemData && itemData.pcf_comparison) {
                // Click to show overlay
                icon.addEventListener('click', (e) => {
                    e.stopPropagation();
                    this.showPcfOverlay(itemData, icon);
                });

                // Hover to show overlay
                icon.addEventListener('mouseenter', () => {
                    this.showPcfOverlay(itemData, icon);
                });
            }
        });
    }

    updateProgressStatus(status) {
        this.showStatus(status, {
            type: 'progress',
            showProgress: true
        });
    }
    
    animateProgressBar() {
        const progressFill = document.querySelector('.carbon-progress-fill');
        if (progressFill) {
            progressFill.style.width = '0%';
            
            // Animate progress
            let progress = 0;
            const interval = setInterval(() => {
                progress += Math.random() * 15;
                if (progress > 90) progress = 90; // Stop at 90% until completion
                progressFill.style.width = progress + '%';
                
                if (progress >= 90) {
                    clearInterval(interval);
                }
            }, 500);
            
            // Store interval for completion
            this.progressInterval = interval;
        }
    }
    
    completeProgress() {
        void 0;
        const progressFill = document.querySelector('.carbon-progress-fill');
        if (progressFill) {
            progressFill.style.width = '100%';
        }
        
        if (this.progressInterval) {
            clearInterval(this.progressInterval);
        }
        
        // Clear status messages when analysis completes to prevent overlap
        void 0;
        this.hideStatus();
    }
    
    async loadRecentItems() {
        void 0;

        const recentItems = document.getElementById('carbon-recent-items');
        if (!recentItems) return;

        try {
            // Use background script to get recent items (it has authentication)
            // Retry logic for service worker reliability (may be terminated after delay)
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            let response = null;
            const maxRetries = 3;

            for (let attempt = 1; attempt <= maxRetries; attempt++) {
                try {
                    response = await runtime.sendMessage({
                        action: 'getRecentItems',
                        limit: 25  // Get more items to have buffer for display
                    });

                    if (response && response.success) {
                        void 0;
                        break;
                    } else if (attempt < maxRetries) {
                        void 0;
                        await new Promise(r => setTimeout(r, 500));
                    }
                } catch (retryError) {
                    void 0;
                    if (attempt < maxRetries) {
                        await new Promise(r => setTimeout(r, 500));
                    }
                }
            }

            if (response && response.success && response.items && response.items.length > 0) {
                // Build HTML directly from the combined individual items + cart summaries
                const itemHtmlPromises = response.items.slice(0, 15).map(async (item, index) => {
                    // Treat any cart/order/email summaries as cart-like for display
                    const summaryTypes = ['cart', 'order', 'email_cart', 'email_order', 'email'];
                    const isCart = summaryTypes.includes(item.item_type) ||
                                   item.analysis_type === 'shopping_cart' ||
                                   item.analysis_type === 'order_page' ||
                                   item.type === 'cart' || item.type === 'order';

                    if (isCart) {
                        // Display cart/order summary from the endpoint
                        const domain = this.extractDomainFromUrl(item.cart_url || item.original_url || '');
                        const date = new Date(item.date_created || item.created_at).toLocaleDateString();

                        const cartSummaryUrl = await this.getCartSummaryUrl(item.session_id, item.analysis_type);
                        const offsetUrl = await this.getOffsetUrl(item.session_id);

                        // Parse the enhanced title to get item count and total
                        let itemCount = item.total_items || 'Unknown';
                        let totalPrice = item.product_total_price ? `$${item.product_total_price.toFixed(2)}` : '';

                        // Add info icon if cart has PCF comparison (same as individual items)
                        const cartInfoIcon = item.pcf_comparison ? ' <span class="carbon-info-icon carbon-recent-info-icon" data-recent-item-id="' + (item.carbonguru_id || item.id) + '" title="View comparable product">ⓘ</span>' : '';

                        // Calculate offset + carbon cost badges for cart (inline with footprint)
                        const cartOffsetCostBadge = item.offset_cost_usd && item.offset_cost_usd > 0
                            ? `<span class="carbon-cost-label">Offset Cost: ${formatCost(item.offset_cost_usd)} USD</span>`
                            : '';
                        // DISABLED: Social cost of carbon display - uncomment to re-enable
                        // const cartCarbonCostBadge = item.social_cost_usd && item.social_cost_usd > 0
                        //     ? `<span class="carbon-cost-label">Carbon Cost: $${parseFloat(item.social_cost_usd).toFixed(2)} USD</span>`
                        //     : '';
                        const cartCarbonCostBadge = ''; // Disabled - social cost not shown

                        return `
                            <div class="carbon-recent-item carbon-cart-item" data-recent-item-id="${item.carbonguru_id || item.id}" data-item-index="${index}">
                                <a href="${encodeURI(cartSummaryUrl)}" target="_blank" class="carbon-recent-name carbon-recent-name-link">🛒 ${this.escapeHtml(item.product_name || `${domain} Cart - ${date}`)}</a>
                                <div class="carbon-recent-footprint">
                                    <span>${item.product_carbon_footprint ? item.product_carbon_footprint.toFixed(1) : 'N/A'} kg CO₂e${cartInfoIcon}</span>
                                </div>
                                <div class="carbon-recent-actions">
                                    <button class="carbon-offset-btn carbon-add-to-basket-btn" data-cart-session="${item.session_id}" data-item-index="${index}" data-preference-type="offset">
                                        <span class="btn-label">Offset:</span>
                                        <span class="btn-cost">...</span>
                                    </button>
                                    <button class="carbon-removal-btn carbon-add-to-basket-btn" data-cart-session="${item.session_id}" data-item-index="${index}" data-preference-type="removal">
                                        <span class="btn-label">Remove:</span>
                                        <span class="btn-cost">...</span>
                                    </button>
                                </div>
                            </div>
                        `;
                    } else {
                        // Display individual item
                        // Check for WooCommerce URL first, fallback to backend URL
                        const productUrl = item.wc_product_url
                            ? item.wc_product_url
                            : (item.details_url ? await this.getProductUrl(item.details_url) : '#');

                        // Debug: Check if PCF comparison data exists
                        void 0;

                        // Add info icon if item has PCF comparison (matches Stage 3 pattern)
                        const infoIcon = item.pcf_comparison ? ' <span class="carbon-info-icon carbon-recent-info-icon" data-recent-item-id="' + (item.carbonguru_id || item.id) + '" title="View comparable product">ⓘ</span>' : '';

                        // Calculate offset + carbon cost if available (inline with footprint)
        const offsetCostBadge = item.offset_cost_usd && item.offset_cost_usd > 0
                            ? `<span class="carbon-cost-label">Offset Cost: ${formatCost(item.offset_cost_usd)} USD</span>`
                            : '';
                        // DISABLED: Social cost of carbon display - uncomment to re-enable
                        // const carbonCostBadge = item.social_cost_usd && item.social_cost_usd > 0
                        //     ? `<span class="carbon-cost-label">Carbon Cost: $${parseFloat(item.social_cost_usd).toFixed(2)} USD</span>`
                        //     : '';
                        const carbonCostBadge = ''; // Disabled - social cost not shown

                        return `
                            <div class="carbon-recent-item" data-recent-item-id="${item.carbonguru_id || item.id}" data-item-index="${index}">
                                <a href="${encodeURI(productUrl)}" target="_blank" class="carbon-recent-name carbon-recent-name-link">${this.escapeHtml(item.product_name || 'Unknown Product')}</a>
                                <div class="carbon-recent-footprint">
                                    <span>${item.product_carbon_footprint ? item.product_carbon_footprint.toFixed(1) : 'N/A'} kg CO₂e${infoIcon}</span>
                                </div>
                                <div class="carbon-recent-actions">
                                    <button class="carbon-offset-btn carbon-add-to-basket-btn" data-item-id="${item.carbonguru_id || item.id}" data-item-index="${index}" data-preference-type="offset">
                                        <span class="btn-label">Offset:</span>
                                        <span class="btn-cost">${item.offset_cost_usd && item.offset_cost_usd > 0 ? formatCost(item.offset_cost_usd) : '...'}</span>
                                    </button>
                                    <button class="carbon-removal-btn carbon-add-to-basket-btn" data-item-id="${item.carbonguru_id || item.id}" data-item-index="${index}" data-preference-type="removal">
                                        <span class="btn-label">Remove:</span>
                                        <span class="btn-cost">${item.removal_cost_usd && item.removal_cost_usd > 0 ? formatCost(item.removal_cost_usd) : (item.offset_cost_usd && item.offset_cost_usd > 0 ? formatCost(parseFloat(item.offset_cost_usd) * 3) : '...')}</span>
                                    </button>
                                </div>
                            </div>
                        `;
                    }
                });
                
                const itemHtmlArray = await Promise.all(itemHtmlPromises);
                recentItems.innerHTML = itemHtmlArray.join('');

                // Store items with PCF comparisons for overlay (including carts)
                this.recentItemsWithComparisons = response.items.filter(item =>
                    item.pcf_comparison
                );

                // Store all items for basket button access (avoid JSON in HTML attributes)
                this.recentItemsData = response.items;

                // Add event listeners for "Add to Basket" buttons
                this.setupBasketButtonListeners();

                // Add event listeners for "View Details" and "Offset" buttons
                this.setupRecentItemsViewAndOffsetButtons();

                // Add event listeners for PCF comparison info icons
                this.setupRecentItemsPcfOverlay();

                // Fetch and display costs for offset/removal buttons
                this.fetchRecentItemCosts();
            } else {
                recentItems.innerHTML = '<div class="carbon-loading">No recent items</div>';
            }
        } catch (error) {
            console.error('CarbonGuru: Error loading recent items:', error);
            recentItems.innerHTML = '<div class="carbon-loading">Error loading items</div>';
        }
    }

    async updatePageAnalysisType() {
        void 0;

        try {
            const currentUrl = window.location.href;
            const pageTitle = document.title;
            const analysisTypeElement = document.getElementById('carbon-page-analysis-type');
            const documentTypeElement = document.getElementById('carbon-document-type');

            if (!analysisTypeElement) {
                void 0;
                return;
            }

            // Send message to background script to check page type
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'detectPageType',
                url: currentUrl,
                pageTitle: pageTitle
            });

            if (!response) {
                analysisTypeElement.textContent = 'Page type: Unknown';
                if (documentTypeElement) documentTypeElement.style.display = 'none';
                return;
            }

            // Handle different page types
            if (response.isOrderPage) {
                analysisTypeElement.textContent = 'Page type: Order';

                // Hide document type (only show page type)
                if (documentTypeElement) {
                    documentTypeElement.style.display = 'none';
                }

                void 0;
                return;
            }

            if (response.isCartPage) {
                analysisTypeElement.textContent = 'Page type: Cart';
                if (documentTypeElement) documentTypeElement.style.display = 'none';
                void 0;
                return;
            }

            if (response.isProductPage) {
                analysisTypeElement.textContent = 'Page type: Product';
                if (documentTypeElement) documentTypeElement.style.display = 'none';
                void 0;
                return;
            }

            if (response.isEmailPage) {
                analysisTypeElement.textContent = 'Page type: Email';
                if (documentTypeElement) documentTypeElement.style.display = 'none';
                void 0;
                return;
            }

            analysisTypeElement.textContent = 'Page type: Unknown';
            if (documentTypeElement) documentTypeElement.style.display = 'none';

        } catch (error) {
            console.error('CarbonGuru: Error updating page analysis type:', error);
        }
    }

    groupItemsBySession(items) {
        const sessionMap = new Map();
        
        // Group items by session_id
        items.forEach(item => {
            if (!sessionMap.has(item.session_id)) {
                sessionMap.set(item.session_id, []);
            }
            sessionMap.get(item.session_id).push(item);
        });
        
        // Convert to array and determine if each session is a cart
        const groups = [];
        sessionMap.forEach((sessionItems, sessionId) => {
            const isCart = sessionId.startsWith('ext_') && sessionItems.length > 1;
            groups.push({
                session_id: sessionId,
                items: sessionItems,
                isCart: isCart
            });
        });
        
        // Sort by most recent first
        groups.sort((a, b) => {
            const aDate = new Date(a.items[0].date_created);
            const bDate = new Date(b.items[0].date_created);
            return bDate - aDate;
        });
        
        return groups.slice(0, 5); // Return top 5 sessions
    }
    
    extractDomainFromUrl(url) {
        if (!url) return 'Unknown';
        try {
            const domain = new URL(url).hostname;
            return domain.replace('www.', '').split('.')[0];
        } catch (e) {
            return 'Unknown';
        }
    }
    
    setupBasketButtonListeners() {
        const basketButtons = document.querySelectorAll('.carbon-add-to-basket-btn');
        basketButtons.forEach(button => {
            button.addEventListener('click', async (e) => {
                e.preventDefault();
                e.stopPropagation();

                // Skip buttons with data-action attribute - these are handled by setupCartCompletionButtons
                // (Analysis Complete status buttons have data-action="add_cart_to_basket")
                if (button.hasAttribute('data-action')) {
                    void 0;
                    return;
                }

                // Get item data from stored reference instead of parsing JSON from attribute
                const itemId = button.getAttribute('data-item-id');
                const cartSession = button.getAttribute('data-cart-session');
                const preferenceType = button.getAttribute('data-preference-type') || 'offset';

                if (!this.recentItemsData) {
                    void 0;
                }

                // Find item by ID or session
                let itemData;
                if (cartSession) {
                    // Cart button - find cart by session_id
                    itemData = this.recentItemsData.find(item =>
                        item.session_id === cartSession && item.item_type === 'cart'
                    );
                } else if (itemId) {
                    // Individual item button - find by carbonguru_id or id
                    itemData = this.recentItemsData.find(item =>
                        (item.carbonguru_id === itemId || item.id === itemId) && item.item_type !== 'cart'
                    );
                }

                // Fallback: use currentAnalysisResult when recent items aren't loaded yet (e.g., Analysis Complete button)
                if (!itemData && this.currentAnalysisResult) {
                    itemData = {
                        ...this.currentAnalysisResult,
                        id: this.currentAnalysisResult.id || this.currentAnalysisResult.carbonguru_id,
                        carbonguru_id: this.currentAnalysisResult.carbonguru_id || this.currentAnalysisResult.id,
                        product_name: this.currentAnalysisResult.product_name || this.currentAnalysisResult.name || 'Unknown Product',
                        product_carbon_footprint: this.currentAnalysisResult.product_carbon_footprint || this.currentAnalysisResult.carbon_footprint || 0,
                        social_cost_usd: this.currentAnalysisResult.social_cost_usd || 0,
                        offset_cost_usd: this.currentAnalysisResult.offset_cost_usd ?? this.currentAnalysisResult.display_cost_usd ?? this.currentAnalysisResult.orderable_cost_usd ?? null,
                        item_type: this.currentAnalysisResult.item_type || 'product',
                        source: 'analysis_complete'
                    };
                }

                if (!itemData) {
                    console.error('CarbonGuru: Item not found for basket add (no recent items and no current analysis result)');
                    button.textContent = 'Error';
                    setTimeout(() => {
                        button.textContent = 'Offset now';
                    }, 2000);
                    return;
                }

                void 0;
                void 0;
                void 0;

                // Check if this is a cart summary - if so, add individual items instead of cart summary
                if (itemData && itemData.item_type === 'cart') {
                    // This is a cart summary - add all individual items
                    void 0;

                    try {
                        // Disable button during request
                        button.disabled = true;
                        button.textContent = 'Adding...';

                        // Validate cart has items
                        if (!itemData.items || !Array.isArray(itemData.items) || itemData.items.length === 0) {
                            console.error('CarbonGuru: Cart has no items:', itemData);
                            button.textContent = 'No Items';
                            setTimeout(() => {
                                button.textContent = 'Offset now';
                                button.disabled = false;
                            }, 2000);
                            return;
                        }

                        // Add each cart item to basket
                        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
                        let successCount = 0;
                        const totalItems = itemData.items.length;

                        for (const item of itemData.items) {
                            // Transform cart item to basket item format
                            const basketItem = {
                                id: item.carbonguru_id || item.id,
                                product_name: item.product_name || item.name || 'Unknown Product',
                                product_carbon_footprint: item.product_carbon_footprint || item.carbon_footprint,
                                social_cost_usd: item.social_cost_usd || 0,
                                product_url: item.product_url || '',
                                product_image: item.product_image || '',
                                details_url: item.details_url || '',
                                session_id: itemData.session_id,
                                preference_type: preferenceType
                            };

                            const response = await runtime.sendMessage({
                                action: 'copyExistingMapping',
                                itemData: basketItem
                            });

                            if (response && response.success) {
                                successCount++;
                            }
                        }

                        // Update button based on success count (no showStatus - keep current display)
                        if (successCount === totalItems) {
                            // Complete success - update button only
                            button.textContent = `✓ Added ${successCount} items`;
                            button.classList.add('carbon-basket-added');
                            button.disabled = true; // Keep disabled to prevent double-add
                            void 0;
                        } else if (successCount > 0) {
                            // Partial success
                            button.textContent = `Added ${successCount}/${totalItems}`;
                            void 0;
                        } else {
                            // Complete failure
                            button.textContent = 'Failed';
                            button.disabled = false;
                            console.error('CarbonGuru: Failed to add any items from cart to basket');
                            // Reset button after 2 seconds for failures only
                            setTimeout(() => {
                                button.textContent = 'Offset now';
                            }, 2000);
                        }

                    } catch (error) {
                        console.error('CarbonGuru: Error adding cart to basket:', error);
                        button.textContent = 'Error';
                        button.disabled = false;

                        // Reset button after 2 seconds
                        setTimeout(() => {
                            button.textContent = 'Offset now';
                        }, 2000);
                    }
                } else {
                    // This is an individual item - use existing behavior
                    try {
                        // Disable button during request
                        button.disabled = true;
                        button.textContent = 'Adding...';

                        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
                        const response = await runtime.sendMessage({
                            action: 'addToBasket',
                            item: { ...itemData, preference_type: preferenceType }
                        });

                        if (response && response.success) {
                            button.textContent = 'Added!';
                            button.classList.add('carbon-basket-added');
                            void 0;

                            // Reset button after 2 seconds
                            setTimeout(() => {
                                button.textContent = 'Offset now';
                                button.classList.remove('carbon-basket-added');
                                button.disabled = false;
                            }, 2000);
                        } else {
                            button.textContent = 'Error';
                            button.disabled = false;
                            console.error('CarbonGuru: Failed to add item to basket:', response);

                            // Reset button after 2 seconds
                            setTimeout(() => {
                                button.textContent = 'Offset now';
                            }, 2000);
                        }
                    } catch (error) {
                        console.error('CarbonGuru: Error adding item to basket:', error);
                        button.textContent = 'Error';
                        button.disabled = false;

                        // Reset button after 2 seconds
                        setTimeout(() => {
                            button.textContent = 'Offset now';
                        }, 2000);
                    }
                }
            });
        });
    }

    // ========== STATUS BUTTON ACTION METHODS ==========

    async addCartToBasket(cartData, preferenceType = 'offset') {
        void 0;

        // Find the Add to basket button to update its state
        const addBtn = document.querySelector('.carbon-add-to-basket-btn[data-action="add_cart_to_basket"]');
        const originalText = addBtn ? addBtn.textContent : 'Offset now';

        // Get preference_type from button if available
        const actualPreferenceType = addBtn?.getAttribute('data-preference-type') || preferenceType;

        try {
            // Update button to show progress
            if (addBtn) {
                addBtn.disabled = true;
                addBtn.textContent = 'Adding...';
            }

            // Get all individual items from the cart session
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;

            if (cartData.items && Array.isArray(cartData.items)) {
                // Add each cart item individually to the basket
                let successCount = 0;
                for (const item of cartData.items) {
                    // Transform cart item to basket item format
                    const basketItem = {
                        id: item.carbonguru_id || item.id,
                        product_name: item.product_name || item.name || 'Unknown Product',
                        product_carbon_footprint: item.product_carbon_footprint || item.carbon_footprint,
                        social_cost_usd: item.social_cost_usd || 0,
                        product_url: item.product_url || '',
                        product_image: item.product_image || '',
                        details_url: item.details_url || '',
                        session_id: cartData.session_id,
                        preference_type: actualPreferenceType
                    };

                    const response = await runtime.sendMessage({
                        action: 'copyExistingMapping',
                        itemData: basketItem
                    });
                    if (response && response.success) {
                        successCount++;
                    }
                }

                // Update button to show success (keep Analysis Complete status visible)
                if (addBtn) {
                    addBtn.textContent = `✓ Added ${successCount} items`;
                    addBtn.classList.add('carbon-basket-added');
                    addBtn.disabled = true; // Keep disabled to prevent double-add
                }
                void 0;
            } else if (cartData.session_id) {
                // Fallback: try to get items from session
                const sessionResponse = await runtime.sendMessage({
                    action: 'getSessionItems',
                    sessionId: cartData.session_id
                });

                if (sessionResponse && sessionResponse.items && sessionResponse.items.length > 0) {
                    let successCount = 0;
                    for (const item of sessionResponse.items) {
                        // Transform cart item to basket item format
                        const basketItem = {
                            id: item.carbonguru_id || item.id,
                            product_name: item.product_name || item.name || 'Unknown Product',
                            product_carbon_footprint: item.product_carbon_footprint || item.carbon_footprint,
                            product_url: item.product_url || '',
                            product_image: item.product_image || '',
                            details_url: item.details_url || '',
                            session_id: cartData.session_id,
                            preference_type: actualPreferenceType
                        };

                        const response = await runtime.sendMessage({
                            action: 'copyExistingMapping',
                            itemData: basketItem
                        });
                        if (response && response.success) {
                            successCount++;
                        }
                    }

                    // Update button to show success (keep Analysis Complete status visible)
                    if (addBtn) {
                        addBtn.textContent = `✓ Added ${successCount} items`;
                        addBtn.classList.add('carbon-basket-added');
                        addBtn.disabled = true; // Keep disabled to prevent double-add
                    }
                    void 0;
                } else {
                    throw new Error('Could not retrieve cart items from session');
                }
            } else {
                throw new Error('No cart items available');
            }
        } catch (error) {
            console.error('CarbonGuru: Error adding cart to basket:', error);
            // Update button to show error (keep Analysis Complete status visible)
            if (addBtn) {
                addBtn.textContent = 'Error - Try Again';
                addBtn.disabled = false;
                // Reset after 3 seconds
                setTimeout(() => {
                    addBtn.textContent = originalText;
                }, 3000);
            }
        }
    }

    async viewCartDetails(cartData) {
        void 0;
        try {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const detailsUrl = await runtime.sendMessage({
                action: 'getCartSummaryUrl',
                sessionId: cartData.session_id,
                analysisType: cartData.analysis_type
            });

            if (detailsUrl && detailsUrl.url) {
                window.open(detailsUrl.url, '_blank');
            } else if (detailsUrl) {
                // Fallback: if detailsUrl is a string (shouldn't happen but be defensive)
                window.open(detailsUrl, '_blank');
            } else {
                // Fallback to details URL from cart data - use correct URL based on analysis type
                const backendUrl = this.environmentConfig?.backendUrl || 'https://carbonguru.io';
                let fallbackUrl;
                if (cartData.analysis_type === 'order_page') {
                    fallbackUrl = cartData.details_url ?
                        `${backendUrl}${cartData.details_url}` :
                        `${backendUrl}/order-summary/order-${cartData.session_id}/`;
                } else {
                    fallbackUrl = cartData.details_url ?
                        `${backendUrl}${cartData.details_url}` :
                        `${backendUrl}/cart-summary/cart-${cartData.session_id}/`;
                }
                window.open(fallbackUrl, '_blank');
            }
        } catch (error) {
            console.error('CarbonGuru: Error opening cart details:', error);
        }
    }

    async offsetCart(cartData) {
        void 0;
        try {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const offsetUrl = await runtime.sendMessage({
                action: 'getOffsetUrl',
                sessionId: cartData.session_id
            });

            if (offsetUrl) {
                window.open(offsetUrl, '_blank');
            } else {
                // Fallback to review URL
                const backendUrl = this.environmentConfig?.backendUrl || 'https://carbonguru.io';
                const fallbackUrl = `${backendUrl}/review/${cartData.session_id}`;
                window.open(fallbackUrl, '_blank');
            }
        } catch (error) {
            console.error('CarbonGuru: Error opening cart offset:', error);
        }
    }

    async addToBasket(productData) {
        void 0;

        // Find the Add to basket button to update its state (for individual items)
        const addBtn = document.querySelector('.carbon-add-to-basket-btn[data-action="add_to_basket"]');
        const originalText = addBtn ? addBtn.textContent : 'Offset now';

        try {
            // Update button to show progress
            if (addBtn) {
                addBtn.disabled = true;
                addBtn.textContent = 'Adding...';
            }

            // Normalize basket payload with required identifiers and costs
            let resolvedId = productData?.id || productData?.carbonguru_id;
            if (!resolvedId && productData?.details_url && productData.details_url.startsWith('/product/')) {
                const parts = productData.details_url.split('/').filter(Boolean);
                resolvedId = parts[parts.length - 1];
            }

            const basketItem = {
                id: resolvedId,
                carbonguru_id: productData?.carbonguru_id || resolvedId,
                product_name: productData?.product_name || productData?.name || productData?.enhanced_display_title || 'Unknown Product',
                product_carbon_footprint: productData?.product_carbon_footprint || productData?.carbon_footprint || 0,
                social_cost_usd: productData?.social_cost_usd || 0,
                offset_cost_usd: productData?.offset_cost_usd ?? productData?.display_cost_usd ?? productData?.orderable_cost_usd ?? null,
                product_url: productData?.product_url || '',
                product_image: productData?.product_image || '',
                details_url: productData?.details_url || '',
                item_type: productData?.item_type || 'product',
                source: productData?.source || 'analysis_complete'
            };

            if (!basketItem.id && !basketItem.carbonguru_id) {
                console.error('CarbonGuru: Cannot add to basket - missing id/carbonguru_id', basketItem);
                if (addBtn) {
                    addBtn.textContent = 'Error - Missing ID';
                    addBtn.disabled = false;
                    setTimeout(() => { addBtn.textContent = originalText; }, 2000);
                }
                return;
            }

            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            await runtime.sendMessage({
                action: 'copyExistingMapping',
                itemData: basketItem
            });

            // Update button to show success (keep Analysis Complete status visible)
            if (addBtn) {
                addBtn.textContent = '✓ Added';
                addBtn.classList.add('carbon-basket-added');
                addBtn.disabled = true; // Keep disabled to prevent double-add
            }
            void 0;
        } catch (error) {
            console.error('CarbonGuru: Error adding product to basket:', error);
            if (addBtn) {
                addBtn.textContent = 'Error - Try Again';
                addBtn.disabled = false;
                setTimeout(() => { addBtn.textContent = originalText; }, 2000);
            }
        }
    }

    async viewDetails(productData) {
        void 0;
        try {
            // If WooCommerce product URL is available, use it directly
            if (productData.wc_product_url) {
                void 0;
                window.open(productData.wc_product_url, '_blank');
                return;
            }

            // Fallback to backend product details page
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const detailsUrl = await runtime.sendMessage({
                action: 'getProductUrl',
                path: productData.details_url
            });

            if (detailsUrl) {
                window.open(detailsUrl, '_blank');
            }
        } catch (error) {
            console.error('CarbonGuru: Error opening product details:', error);
        }
    }

    async offsetNow(productData) {
        void 0;
        try {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const offsetUrl = await runtime.sendMessage({
                action: 'getReviewUrl',
                detailsUrl: productData.details_url
            });

            if (offsetUrl) {
                window.open(offsetUrl, '_blank');
            }
        } catch (error) {
            console.error('CarbonGuru: Error opening product offset:', error);
        }
    }

    // Show original text description extracted status
    showTextExtractionStatus(data) {
        void 0;

        const statusSection = document.getElementById('carbon-status-section');
        const statusText = document.getElementById('carbon-status-text');
        const statusIcon = document.getElementById('carbon-status-icon');
        const resultsSection = document.getElementById('carbon-results-section');

        if (!statusText) return;

        // Clear any previous results section content to prevent stacking
        if (resultsSection) {
            resultsSection.style.display = 'none';
        }

        // Hide the status icon for card display (matching cart extraction style)
        if (statusIcon) {
            statusIcon.style.display = 'none';
        }

        // Show the status section
        statusSection.style.display = 'block';
        statusSection.className = 'carbon-status-section carbon-status-success';

        const structuredData = data.structured_content || {};
        const originalText = structuredData.original_text || 'Product text extracted';
        const textLength = structuredData.text_length || 0;
        const itemIndex = structuredData.item_index || '';

        // Create card-style content matching other status messages
        const statusHtml = `
            <div class="carbon-status-card">
                <div class="carbon-status-header">
                    <span class="carbon-status-stage">${itemIndex ? `Item ${itemIndex}: ` : ''}Initial description</span>
                </div>
                <div class="carbon-text-preview">${this.escapeHtml(originalText)}</div>
                <div class="carbon-text-meta">Length: ${textLength} characters</div>
            </div>
        `;

        statusText.innerHTML = statusHtml;
    }

    // Show cleaned text description processed status
    showTextCleaningStatus(data) {
        void 0;

        const statusSection = document.getElementById('carbon-status-section');
        const statusText = document.getElementById('carbon-status-text');
        const statusIcon = document.getElementById('carbon-status-icon');
        const resultsSection = document.getElementById('carbon-results-section');

        if (!statusText) return;

        // Clear any previous results section content to prevent stacking
        if (resultsSection) {
            resultsSection.style.display = 'none';
        }

        // Hide the status icon for card display (matching cart extraction style)
        if (statusIcon) {
            statusIcon.style.display = 'none';
        }

        // Show the status section
        statusSection.style.display = 'block';
        statusSection.className = 'carbon-status-section carbon-status-success';

        const structuredData = data.structured_content || {};
        const cleanedText = structuredData.cleaned_text || 'Product text cleaned';
        const originalLength = structuredData.original_length || 0;
        const cleanedLength = structuredData.cleaned_length || 0;
        const itemIndex = structuredData.item_index || '';

        // Create card-style content matching other status messages
        const statusHtml = `
            <div class="carbon-status-card">
                <div class="carbon-status-header">
                    <span class="carbon-status-stage">${itemIndex ? `Item ${itemIndex}: ` : ''}Final description</span>
                </div>
                <div class="carbon-text-preview">${this.escapeHtml(cleanedText)}</div>
                <div class="carbon-text-meta">Original: ${originalLength} → Cleaned: ${cleanedLength} characters</div>
            </div>
        `;

        statusText.innerHTML = statusHtml;
    }

    // Show product extraction status (LLM extracted product info)
    showProductExtractionStatus(data) {
        void 0;

        const statusSection = document.getElementById('carbon-status-section');
        const statusText = document.getElementById('carbon-status-text');
        const statusIcon = document.getElementById('carbon-status-icon');
        const resultsSection = document.getElementById('carbon-results-section');

        if (!statusText) return;

        // Clear any previous results section content to prevent stacking
        if (resultsSection) {
            resultsSection.style.display = 'none';
        }

        // Hide the status icon for card display (matching cart extraction style)
        if (statusIcon) {
            statusIcon.style.display = 'none';
        }

        // Show the status section
        statusSection.style.display = 'block';
        statusSection.className = 'carbon-status-section carbon-status-success';

        const structuredData = data.structured_content || {};
        const productName = structuredData.product_name || 'Product';
        const price = structuredData.price || 0;
        const currency = structuredData.currency || 'USD';
        const category = structuredData.category || 'N/A';
        const url = structuredData.url || window.location.href;
        const extractionTime = structuredData.extraction_time || data.timestamp;

        // Format price with currency symbol
        const currencySymbol = this.getCurrencySymbol(currency);
        const formattedPrice = typeof price === 'number' ? price.toFixed(2) : price;

        // Format timestamp
        let formattedTime = '';
        if (extractionTime) {
            const date = new Date(extractionTime);
            formattedTime = date.toLocaleString('en-US', {
                month: 'short',
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            });
        }

        // Truncate URL for display (show domain + path start)
        let displayUrl = url;
        try {
            const urlObj = new URL(url);
            const path = urlObj.pathname.length > 30 ? urlObj.pathname.substring(0, 30) + '...' : urlObj.pathname;
            displayUrl = urlObj.hostname + path;
        } catch (e) {
            displayUrl = url.length > 50 ? url.substring(0, 50) + '...' : url;
        }

        // Create card-style content matching other status messages
        const statusHtml = `
            <div class="carbon-status-card">
                <div class="carbon-status-header">
                    <span class="carbon-status-stage">📦 Product Extracted</span>
                </div>
                <div class="carbon-extraction-meta">
                    <div class="carbon-extraction-url" title="${this.escapeHtml(url)}">${this.escapeHtml(displayUrl)}</div>
                    <div class="carbon-extraction-time">${formattedTime}</div>
                </div>
                <div class="carbon-product-info">
                    <div class="carbon-product-name">${this.escapeHtml(productName)}</div>
                    <div class="carbon-product-price">${currencySymbol}${formattedPrice}</div>
                </div>
            </div>
        `;

        statusText.innerHTML = statusHtml;
    }

    // Show learning triggered status (new domain pattern learning)
    showLearningTriggeredStatus(data) {
        void 0;

        const statusSection = document.getElementById('carbon-status-section');
        const statusText = document.getElementById('carbon-status-text');
        const statusIcon = document.getElementById('carbon-status-icon');
        const resultsSection = document.getElementById('carbon-results-section');

        if (!statusText) return;

        // Clear any previous results section content to prevent stacking
        if (resultsSection) {
            resultsSection.style.display = 'none';
        }

        // Hide the status icon for card display (matching cart extraction style)
        if (statusIcon) {
            statusIcon.style.display = 'none';
        }

        // Show the status section
        statusSection.style.display = 'block';
        statusSection.className = 'carbon-status-section carbon-status-info';

        const structuredData = data.structured_content || {};
        const domain = structuredData.domain || 'this site';
        const learningType = structuredData.learning_type || 'extraction';

        // Create card-style content matching other status messages
        const statusHtml = `
            <div class="carbon-status-card carbon-learning-card">
                <div class="carbon-status-header">
                    <span class="carbon-status-stage">🎓 Learning Mode</span>
                </div>
                <div class="carbon-learning-info">
                    <div class="carbon-learning-domain">Training on: <strong>${this.escapeHtml(domain)}</strong></div>
                    <div class="carbon-learning-type">Type: ${learningType} patterns</div>
                </div>
                <div class="carbon-text-meta">Learning helps improve CarbonGuru's speed and accuracy in the future.</div>
            </div>
        `;

        statusText.innerHTML = statusHtml;
    }

    /**
     * Show Analysis Complete status for individual products
     * Called when individual_item_complete is received for a non-cart product
     */
    showAnalysisComplete(data) {
        void 0;

        // Complete any progress indicators
        this.completeProgress();

        // Show the results using the standard showResults function
        // This displays the green expandable tile with all product data
        this.showResults(data);
    }

    // ========== GMAIL EMAIL PROCESSING METHODS ==========

    /**
     * Get auth header for Gmail API requests
     * Used by GmailIntegration class
     */
    async getAuthHeaderForGmail() {
        return new Promise((resolve) => {
            browser.runtime.sendMessage({ action: 'getAuthHeader' }, (response) => {
                if (response && response.authHeader) {
                    resolve(response.authHeader);
                } else {
                    resolve(null);
                }
            });
        });
    }

    /**
     * Process single Gmail email (email view page)
     */
    async processGmailSingleEmail() {
        void 0;

        try {
            // Check Gmail connection first (same as batch path)
            const connectionStatus = await this.gmailIntegration.checkGmailConnection();

            if (!connectionStatus.connected) {
                const backendUrl = this.environmentConfig?.backendUrl || 'https://carbonguru.io';
                const settingsUrl = `${backendUrl}/my-account/gmail-settings/`;
                this.showGmailConnectionError(settingsUrl);
                return;
            }

            // Get email ID from URL
            const emailId = this.gmailDetector.getCurrentEmailId();

            if (!emailId) {
                throw new Error('Could not extract email ID from Gmail page');
            }

            // Show status
            this.showStatus('Processing email...', { type: 'info' });

            // Process email via Gmail integration
            const result = await this.gmailIntegration.processSingleEmail(
                emailId,
                (status) => this.handleGmailStatus(status)
            );

            if (!result.success) {
                throw new Error(result.error || 'Email processing failed');
            }

            // Show result
            if (result.document) {
                this.showGmailDocumentResult(result.document);
            }

            // Reload recent items
            this.loadRecentItems();

        } catch (error) {
            console.error('CarbonGuru Gmail: Single email processing error:', error);
            // Check if this is a Gmail not connected error
            if (error.message && error.message.includes('Gmail not connected')) {
                const backendUrl = this.environmentConfig?.backendUrl || 'https://carbonguru.io';
                const settingsUrl = `${backendUrl}/my-account/gmail-settings/`;
                this.showGmailConnectionError(settingsUrl);
            } else {
                this.showStatus(error.message, { type: 'error', duration: 5000 });
            }
        }
    }

    /**
     * Process Gmail batch emails (inbox/list view page)
     * Shows scope selector
     */
    async processGmailBatchEmails() {
        void 0;

        try {
            // Check Gmail connection first
            const connectionStatus = await this.gmailIntegration.checkGmailConnection();

            if (!connectionStatus.connected) {
                // Show error with link to Gmail settings page
                const backendUrl = this.environmentConfig?.backendUrl || 'https://carbonguru.io';
                const settingsUrl = `${backendUrl}/my-account/gmail-settings/`;
                this.showGmailConnectionError(settingsUrl);
                return;
            }

            // Clear any previous status messages (e.g., stale "Gmail not connected" error)
            this.hideStatus();

            // Show scope selector via floating tile extension
            if (this.gmailFloatingTile) {
                this.gmailFloatingTile.showScopeSelector();
            } else {
                throw new Error('Gmail floating tile extension not initialized');
            }

        } catch (error) {
            console.error('CarbonGuru Gmail: Batch email processing error:', error);
            this.showStatus(error.message, { type: 'error', duration: 5000 });
        }
    }

    /**
     * Show Gmail connection error with button to settings page
     * Uses same button style as WordPress login prompt
     */
    showGmailConnectionError(settingsUrl) {
        const buttons = [
            { text: 'Connect Gmail', class: 'success', action: 'connect-gmail' },
            { text: 'Dismiss', class: 'secondary', action: 'dismiss' }
        ];

        this.showStatus(
            'Gmail not connected. Please connect your Gmail account to analyze purchase emails.',
            {
                type: 'error',
                interactive: true,
                buttons: buttons,
                duration: 0 // Don't auto-dismiss
            }
        );

        // Handle button clicks
        const statusButtons = document.getElementById('carbon-status-buttons');
        if (statusButtons) {
            const connectButton = statusButtons.querySelector('[data-action="connect-gmail"]');
            const dismissButton = statusButtons.querySelector('[data-action="dismiss"]');

            if (connectButton) {
                connectButton.onclick = () => {
                    void 0;
                    window.open(settingsUrl, '_blank');
                    this.hideStatus();
                };
            }

            if (dismissButton) {
                dismissButton.onclick = () => {
                    void 0;
                    this.hideStatus();
                };
            }
        }
    }

    /**
     * Handle Gmail status updates from integration
     */
    handleGmailStatus(status) {
        void 0;

        if (status.stage === 'document_complete' && status.document) {
            // Stage 1: Email document extracted - show enhanced title
            const doc = status.document;
            const message = `${doc.enhanced_title || 'Email processed'}`;
            this.addStatusMessage(message, { type: 'success' });

        } else if (status.stage === 'batch_processing') {
            // Batch progress
            const message = `Processing email ${status.current} of ${status.total}...`;
            this.showStatus(message, { type: 'info' });

        } else if (status.type === 'error') {
            // Error
            this.showStatus(status.message, { type: 'error', duration: 5000 });

        } else {
            // Generic status
            this.showStatus(status.message, { type: status.type || 'info' });
        }
    }

    /**
     * Show Gmail document result in floating tile
     */
    showGmailDocumentResult(document) {
        void 0;

        // Format as enhanced title message
        const title = document.enhanced_title || `Email • ${document.items_count || 0} items`;
        const carbon = document.total_carbon ? `${document.total_carbon.toFixed(2)} kg CO₂e` : '';
        const cost = document.total_cost ? `$${document.total_cost.toFixed(2)}` : '';

        const message = `${title}${carbon ? ' • ' + carbon : ''}${cost ? ' • ' + cost : ''}`;

        this.showStatus(message, { type: 'success', duration: 10000 });
    }

    /**
     * Add status message to floating tile (for multi-message display)
     */
    addStatusMessage(message, options = {}) {
        // For now, just use showStatus
        // In future, could implement message queue/stack
        this.showStatus(message, options);
    }

}


// Shopping cart data collection removed - extension only does data acquisition
// Backend handles all cart vs product detection and processing decisions

// Initialize the content script
new CarbonGuruContentScript();
